<?php

namespace Drupal\sphoenix_ai\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Access\AccessResult;
use Drupal\Core\Session\AccountInterface;
use Drupal\sphoenix_ai\Service\SubscriptionService;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\JsonResponse;

/**
 * Controller for subscription management endpoints.
 */
class SubscriptionController extends ControllerBase
{

  /**
   * The subscription service.
   *
   * @var \Drupal\sphoenix_ai\Service\SubscriptionService
   */
  protected $subscriptionService;

  /**
   * Constructs a SubscriptionController object.
   *
   * @param \Drupal\sphoenix_ai\Service\SubscriptionService $subscription_service
   *   The subscription service.
   */
  public function __construct(SubscriptionService $subscription_service)
  {
    $this->subscriptionService = $subscription_service;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container)
  {
    return new static(
      $container->get('sphoenix_ai.subscription')
    );
  }

  /**
   * Get current subscription status.
   *
   * @return \Symfony\Component\HttpFoundation\JsonResponse
   *   The JSON response.
   */
  public function status()
  {
    try {
      $usage_info = $this->subscriptionService->getUsageInfo();
      $subscription_status = $this->subscriptionService->formatSubscriptionStatus();

      return new JsonResponse([
        'success' => TRUE,
        'data' => [
          'subscription' => $this->subscriptionService->getCurrentSubscription(),
          'usage_info' => $usage_info,
          'status' => $subscription_status,
          'has_active_subscription' => $this->subscriptionService->hasActiveSubscription(),
          'has_usage_remaining' => $this->subscriptionService->hasUsageRemaining(),
          'current_plan_name' => $this->subscriptionService->getCurrentPlanName(),
          'plans_url' => $this->subscriptionService->getPlansUrl(),
          'recommendations' => $this->subscriptionService->getUsageRecommendations(),
        ],
      ]);
    } catch (\Exception $e) {
      return new JsonResponse([
        'success' => FALSE,
        'message' => 'Error getting subscription status: ' . $e->getMessage(),
        'data' => [
          'subscription' => NULL,
          'usage_info' => NULL,
          'status' => [
            'status' => 'error',
            'message' => 'Unable to load subscription information',
          ],
          'has_active_subscription' => FALSE,
          'has_usage_remaining' => FALSE,
          'current_plan_name' => 'Unknown',
          'plans_url' => '#',
          'recommendations' => [],
        ],
      ], 500);
    }
  }

  /**
   * Get available subscription plans.
   *
   * @return \Symfony\Component\HttpFoundation\JsonResponse
   *   The JSON response.
   */
  public function plans()
  {
    try {
      $plans = $this->subscriptionService->getAvailablePlans();
      $current_subscription = $this->subscriptionService->getCurrentSubscription();
      $current_plan_name = $this->subscriptionService->getCurrentPlanName();

      return new JsonResponse([
        'success' => TRUE,
        'data' => [
          'available_plans' => $plans,
          'current_plan' => $current_plan_name,
          'has_subscription' => $this->subscriptionService->hasActiveSubscription(),
          'plans_url' => $this->subscriptionService->getPlansUrl(),
          'upgrade_recommendations' => $this->getUpgradeRecommendations($current_subscription, $plans),
        ],
      ]);
    } catch (\Exception $e) {
      return new JsonResponse([
        'success' => FALSE,
        'message' => 'Error getting subscription plans: ' . $e->getMessage(),
      ], 500);
    }
  }

  /**
   * Check if user can use a specific feature.
   *
   * @param string $feature
   *   The feature name.
   *
   * @return \Symfony\Component\HttpFoundation\JsonResponse
   *   The JSON response.
   */
  public function canUseFeature($feature)
  {
    try {
      $can_use = $this->subscriptionService->canUseFeature($feature);
      $reason = NULL;

      if (!$can_use) {
        $subscription_status = $this->subscriptionService->formatSubscriptionStatus();
        $reason = $subscription_status['message'];
      }

      return new JsonResponse([
        'success' => TRUE,
        'data' => [
          'can_use_feature' => $can_use,
          'feature' => $feature,
          'reason' => $reason,
          'remaining_usage' => $this->subscriptionService->hasUsageRemaining() ?
            $this->subscriptionService->getCurrentSubscription()['total_remaining_usage'] ?? 0 : 0,
          'plans_url' => $this->subscriptionService->getPlansUrl(),
        ],
      ]);
    } catch (\Exception $e) {
      return new JsonResponse([
        'success' => FALSE,
        'message' => 'Error checking feature access: ' . $e->getMessage(),
        'data' => [
          'can_use_feature' => FALSE,
          'feature' => $feature,
          'reason' => 'Service error',
        ],
      ], 500);
    }
  }

  /**
   * Get upgrade recommendations based on current usage.
   *
   * @param array|null $current_subscription
   *   Current subscription data.
   * @param array $available_plans
   *   Available plans.
   *
   * @return array
   *   Upgrade recommendations.
   */
  protected function getUpgradeRecommendations($current_subscription, array $available_plans)
  {
    $recommendations = [];

    if (!$current_subscription) {
      return [
        [
          'type' => 'get_started',
          'title' => 'Get Started with AI',
          'description' => 'Choose a plan to start using AI-powered content tools',
          'recommended_plan' => 'starter',
          'reason' => 'Most popular for individual creators',
        ],
      ];
    }

    $usage_info = $this->subscriptionService->getUsageInfo();
    if (!$usage_info) {
      return [];
    }

    $current_plan_name = strtolower($usage_info['plan_name']);
    $usage_percentage = $usage_info['usage_percentage'];

    // High usage recommendations
    if ($usage_percentage > 80) {
      if ($current_plan_name === 'starter') {
        $recommendations[] = [
          'type' => 'upgrade',
          'title' => 'Upgrade to Pro',
          'description' => 'You are using ' . round($usage_percentage) . '% of your monthly limit',
          'recommended_plan' => 'pro',
          'reason' => '5x more requests + priority processing',
        ];
      } elseif ($current_plan_name === 'pro') {
        $recommendations[] = [
          'type' => 'upgrade',
          'title' => 'Consider Enterprise',
          'description' => 'You are using ' . round($usage_percentage) . '% of your monthly limit',
          'recommended_plan' => 'enterprise',
          'reason' => 'Unlimited requests + advanced features',
        ];
      } else {
        $recommendations[] = [
          'type' => 'add_subscription',
          'title' => 'Add More Plans',
          'description' => 'Stack multiple subscriptions for more usage',
          'recommended_plan' => $current_plan_name,
          'reason' => 'Combine plans for maximum flexibility',
        ];
      }
    }

    // Feature-based recommendations
    if ($current_plan_name === 'free') {
      $recommendations[] = [
        'type' => 'upgrade',
        'title' => 'Unlock Full Features',
        'description' => 'Access 100+ AI tools and advanced features',
        'recommended_plan' => 'starter',
        'reason' => 'Best value for comprehensive AI toolkit',
      ];
    }

    // Multiple subscription benefits
    if ($current_subscription['active_subscriptions_count'] == 1) {
      $recommendations[] = [
        'type' => 'stack_plans',
        'title' => 'Stack Multiple Plans',
        'description' => 'Combine different plans for maximum usage',
        'recommended_plan' => 'mixed',
        'reason' => 'Usage limits combine across all active plans',
      ];
    }

    return $recommendations;
  }

  /**
   * Record usage for a feature.
   *
   * @param string $feature
   *   The feature name.
   *
   * @return \Symfony\Component\HttpFoundation\JsonResponse
   *   The JSON response.
   */
  public function recordUsage($feature)
  {
    try {
      $this->subscriptionService->recordUsage($feature, 1);

      $usage_info = $this->subscriptionService->getUsageInfo();

      return new JsonResponse([
        'success' => TRUE,
        'data' => [
          'feature' => $feature,
          'usage_recorded' => TRUE,
          'remaining_usage' => $usage_info['remaining_usage'] ?? 0,
          'usage_percentage' => $usage_info['usage_percentage'] ?? 0,
        ],
      ]);
    } catch (\Exception $e) {
      return new JsonResponse([
        'success' => FALSE,
        'message' => 'Error recording usage: ' . $e->getMessage(),
      ], 500);
    }
  }

  /**
   * Access callback for subscription endpoints.
   *
   * @param \Drupal\Core\Session\AccountInterface $account
   *   The user account.
   *
   * @return \Drupal\Core\Access\AccessResultInterface
   *   The access result.
   */
  public function access(AccountInterface $account)
  {
    return AccessResult::allowedIfHasPermission($account, 'use ai content assistant');
  }
}
