<?php

namespace Drupal\sphoenix_ai\Form;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Config\TypedConfigManagerInterface;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\sphoenix_ai\Service\ApiClientService;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Configure SPhoenix AI settings form.
 */
class SettingsForm extends ConfigFormBase
{

  /**
   * The API client service.
   *
   * @var \Drupal\sphoenix_ai\Service\ApiClientService
   */
  protected $apiClient;

  /**
   * Constructs a SettingsForm object.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The factory for configuration objects.
   * @param \Drupal\Core\Config\TypedConfigManagerInterface $typed_config_manager
   *   The typed configuration manager.
   * @param \Drupal\sphoenix_ai\Service\ApiClientService $api_client
   *   The API client service.
   */
  public function __construct(ConfigFactoryInterface $config_factory, TypedConfigManagerInterface $typed_config_manager, ApiClientService $api_client)
  {
    parent::__construct($config_factory, $typed_config_manager);
    $this->apiClient = $api_client;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container)
  {
    return new static(
      $container->get('config.factory'),
      $container->get('config.typed'),
      $container->get('sphoenix_ai.api_client')
    );
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames()
  {
    return ['sphoenix_ai.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId()
  {
    return 'sphoenix_ai_settings_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state)
  {
    $config = $this->config('sphoenix_ai.settings');

    $form['#tree'] = TRUE;

    // Default Configuration Values
    $form['defaults'] = [
      '#type' => 'details',
      '#title' => $this->t('Default Configuration'),
      '#description' => $this->t('These values will be used as defaults for AI requests but can be overridden per request.'),
      '#open' => TRUE,
    ];

    $form['defaults']['default_temperature'] = [
      '#type' => 'number',
      '#title' => $this->t('Default Temperature'),
      '#description' => $this->t('Controls randomness in AI responses (0.0-1.0). Lower values make responses more focused and deterministic.'),
      '#default_value' => $config->get('default_temperature') ?: 0.7,
      '#min' => 0,
      '#max' => 1,
      '#step' => 0.1,
    ];

    $form['defaults']['default_tone'] = [
      '#type' => 'select',
      '#title' => $this->t('Default Tone'),
      '#description' => $this->t('The default tone for generated content'),
      '#default_value' => $config->get('default_tone') ?: 'conversational',
      '#options' => [
        'professional' => $this->t('Professional'),
        'conversational' => $this->t('Conversational'),
        'casual' => $this->t('Casual'),
        'formal' => $this->t('Formal'),
        'friendly' => $this->t('Friendly'),
        'authoritative' => $this->t('Authoritative'),
        'persuasive' => $this->t('Persuasive'),
        'informative' => $this->t('Informative'),
      ],
    ];

    $form['defaults']['default_word_length'] = [
      '#type' => 'number',
      '#title' => $this->t('Default Word Length'),
      '#description' => $this->t('Default target word count for generated content'),
      '#default_value' => $config->get('default_word_length') ?: 600,
      '#min' => 50,
      '#max' => 5000,
      '#step' => 50,
    ];

    $form['defaults']['default_paragraphs'] = [
      '#type' => 'number',
      '#title' => $this->t('Default Paragraphs'),
      '#description' => $this->t('Default number of paragraphs for generated content'),
      '#default_value' => $config->get('default_paragraphs') ?: 4,
      '#min' => 1,
      '#max' => 20,
    ];

    $form['defaults']['include_summary'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Include Summary by Default'),
      '#description' => $this->t('Include a summary/TL;DR section in generated content by default'),
      '#default_value' => $config->get('include_summary') ?? TRUE,
    ];

    $form['defaults']['include_faq'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Include FAQ by Default'),
      '#description' => $this->t('Include an FAQ section in generated content by default'),
      '#default_value' => $config->get('include_faq') ?? FALSE,
    ];

    // Image Generation Defaults
    $form['defaults']['default_image_width'] = [
      '#type' => 'number',
      '#title' => $this->t('Default Image Width'),
      '#description' => $this->t('Default width for generated images'),
      '#default_value' => $config->get('default_image_width') ?: 1024,
      '#min' => 256,
      '#max' => 2048,
      '#step' => 64,
    ];

    $form['defaults']['default_image_height'] = [
      '#type' => 'number',
      '#title' => $this->t('Default Image Height'),
      '#description' => $this->t('Default height for generated images'),
      '#default_value' => $config->get('default_image_height') ?: 1024,
      '#min' => 256,
      '#max' => 2048,
      '#step' => 64,
    ];

    // SEO & Content Configuration
    $form['seo'] = [
      '#type' => 'details',
      '#title' => $this->t('SEO & Content Configuration'),
      '#open' => FALSE,
    ];

    $form['seo']['default_keywords'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Default Keywords'),
      '#description' => $this->t('Enter default keywords, one per line. These will be used for content optimization.'),
      '#default_value' => is_array($config->get('default_keywords')) ? implode("\n", $config->get('default_keywords')) : '',
      '#rows' => 5,
    ];

    $form['seo']['competitor_urls'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Competitor URLs'),
      '#description' => $this->t('Enter competitor URLs, one per line. These will be used for competitive analysis.'),
      '#default_value' => is_array($config->get('competitor_urls')) ? implode("\n", $config->get('competitor_urls')) : '',
      '#rows' => 5,
    ];

    // Humanization Settings
    $form['humanization'] = [
      '#type' => 'details',
      '#title' => $this->t('AI Humanization Settings'),
      '#open' => FALSE,
    ];

    $form['humanization']['enable_humanization'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable Humanization'),
      '#description' => $this->t('Apply humanization techniques to make AI-generated content appear more natural'),
      '#default_value' => $config->get('enable_humanization') ?? TRUE,
    ];

    $form['humanization']['humanization_level'] = [
      '#type' => 'select',
      '#title' => $this->t('Humanization Level'),
      '#description' => $this->t('The level of humanization to apply'),
      '#default_value' => $config->get('humanization_level') ?: 'medium',
      '#options' => [
        'light' => $this->t('Light'),
        'medium' => $this->t('Medium'),
        'heavy' => $this->t('Heavy'),
      ],
      '#states' => [
        'visible' => [
          ':input[name="humanization[enable_humanization]"]' => ['checked' => TRUE],
        ],
      ],
    ];

    // Chatbot Interface Configuration
    $form['interface'] = [
      '#type' => 'details',
      '#title' => $this->t('Chatbot Interface Configuration'),
      '#open' => FALSE,
    ];

    $form['interface']['chatbot_position'] = [
      '#type' => 'select',
      '#title' => $this->t('Chatbot Position'),
      '#description' => $this->t('Where to position the chatbot on the page'),
      '#default_value' => $config->get('chatbot_position') ?: 'right',
      '#options' => [
        'right' => $this->t('Right side'),
        'left' => $this->t('Left side'),
        'bottom' => $this->t('Bottom'),
      ],
    ];

    $form['interface']['chatbot_theme'] = [
      '#type' => 'select',
      '#title' => $this->t('Chatbot Theme'),
      '#description' => $this->t('Visual theme for the chatbot interface'),
      '#default_value' => $config->get('chatbot_theme') ?: 'default',
      '#options' => [
        'default' => $this->t('Default'),
        'dark' => $this->t('Dark'),
        'light' => $this->t('Light'),
        'minimal' => $this->t('Minimal'),
      ],
    ];

    $form['interface']['auto_expand_chatbot'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Auto-expand Chatbot'),
      '#description' => $this->t('Automatically expand the chatbot when the page loads'),
      '#default_value' => $config->get('auto_expand_chatbot') ?? FALSE,
    ];

    $form['interface']['auto_suggest_tasks'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Auto-suggest Tasks'),
      '#description' => $this->t('Show AI task suggestions based on the focused field'),
      '#default_value' => $config->get('auto_suggest_tasks') ?? TRUE,
    ];

    // Usage & Performance
    $form['performance'] = [
      '#type' => 'details',
      '#title' => $this->t('Usage & Performance'),
      '#open' => FALSE,
    ];

    $form['performance']['cleanup_usage_days'] = [
      '#type' => 'number',
      '#title' => $this->t('Usage Data Retention (Days)'),
      '#description' => $this->t('Number of days to keep usage statistics. Set to 0 to keep forever.'),
      '#default_value' => $config->get('cleanup_usage_days') ?: 365,
      '#min' => 0,
      '#max' => 9999,
    ];

    $form['performance']['cache_responses'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Cache API Responses'),
      '#description' => $this->t('Cache API responses to improve performance and reduce costs'),
      '#default_value' => $config->get('cache_responses') ?? TRUE,
    ];

    $form['performance']['cache_duration'] = [
      '#type' => 'number',
      '#title' => $this->t('Cache Duration (Hours)'),
      '#description' => $this->t('How long to cache API responses'),
      '#default_value' => $config->get('cache_duration') ?: 1,
      '#min' => 0.1,
      '#max' => 168,
      '#step' => 0.1,
      '#states' => [
        'visible' => [
          ':input[name="performance[cache_responses]"]' => ['checked' => TRUE],
        ],
      ],
    ];

    return parent::buildForm($form, $form_state);
  }


  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state)
  {
    parent::validateForm($form, $form_state);

    // Validate temperature range
    $temperature = $form_state->getValue(['defaults', 'default_temperature']);
    if ($temperature < 0 || $temperature > 1) {
      $form_state->setErrorByName('defaults][default_temperature', $this->t('Temperature must be between 0.0 and 1.0.'));
    }
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state)
  {
    $config = $this->config('sphoenix_ai.settings');
    // Default Configuration
    $config->set('default_temperature', (float) $form_state->getValue(['defaults', 'default_temperature']));
    $config->set('default_tone', $form_state->getValue(['defaults', 'default_tone']));
    $config->set('default_word_length', (int) $form_state->getValue(['defaults', 'default_word_length']));
    $config->set('default_paragraphs', (int) $form_state->getValue(['defaults', 'default_paragraphs']));
    $config->set('include_summary', (bool) $form_state->getValue(['defaults', 'include_summary']));
    $config->set('include_faq', (bool) $form_state->getValue(['defaults', 'include_faq']));
    $config->set('default_image_width', (int) $form_state->getValue(['defaults', 'default_image_width']));
    $config->set('default_image_height', (int) $form_state->getValue(['defaults', 'default_image_height']));

    // SEO Configuration
    $keywords_text = $form_state->getValue(['seo', 'default_keywords']);
    $keywords = array_filter(array_map('trim', explode("\n", $keywords_text)));
    $config->set('default_keywords', $keywords);

    $competitor_urls_text = $form_state->getValue(['seo', 'competitor_urls']);
    $competitor_urls = array_filter(array_map('trim', explode("\n", $competitor_urls_text)));
    $config->set('competitor_urls', $competitor_urls);

    // Humanization Configuration
    $config->set('enable_humanization', (bool) $form_state->getValue(['humanization', 'enable_humanization']));
    $config->set('humanization_level', $form_state->getValue(['humanization', 'humanization_level']));

    // Interface Configuration
    $config->set('chatbot_position', $form_state->getValue(['interface', 'chatbot_position']));
    $config->set('chatbot_theme', $form_state->getValue(['interface', 'chatbot_theme']));
    $config->set('auto_expand_chatbot', (bool) $form_state->getValue(['interface', 'auto_expand_chatbot']));
    $config->set('auto_suggest_tasks', (bool) $form_state->getValue(['interface', 'auto_suggest_tasks']));

    // Performance Configuration
    $config->set('cleanup_usage_days', (int) $form_state->getValue(['performance', 'cleanup_usage_days']));
    $config->set('cache_responses', (bool) $form_state->getValue(['performance', 'cache_responses']));
    $config->set('cache_duration', (float) $form_state->getValue(['performance', 'cache_duration']));

    $config->save();

    // Clear API cache after settings change
    $this->apiClient->clearCache();

    $this->messenger()->addMessage($this->t('SPhoenix AI settings have been saved.'));

    parent::submitForm($form, $form_state);
  }
}
