<?php

namespace Drupal\sphoenix_ai\Service;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Session\AccountProxyInterface;
use Drupal\Core\TempStore\PrivateTempStoreFactory;
use Drupal\Core\TempStore\PrivateTempStore;
use Drupal\Core\Http\ClientFactory;
use Drupal\Core\Logger\LoggerChannelInterface;
use Drupal\Core\Url;
use GuzzleHttp\Client;
use GuzzleHttp\Exception\GuzzleException;

/**
 * Service for handling SPhoenix AI authentication (VS Code pattern).
 */
class AuthenticationService
{

  /**
   * The HTTP client.
   *
   * @var \GuzzleHttp\Client
   */
  protected $httpClient;

  /**
   * The private tempstore.
   *
   * @var \Drupal\Core\TempStore\PrivateTempStore
   */
  protected $tempstore;

  /**
   * The current user.
   *
   * @var \Drupal\Core\Session\AccountProxyInterface
   */
  protected $currentUser;

  /**
   * The logger.
   *
   * @var \Drupal\Core\Logger\LoggerChannelInterface
   */
  protected $logger;

  /**
   * Constructs an AuthenticationService object.
   */
  public function __construct(
    ClientFactory $http_client_factory,
    PrivateTempStoreFactory $tempstore_factory,
    AccountProxyInterface $current_user,
    LoggerChannelInterface $logger
  ) {
    $this->httpClient = $http_client_factory->fromOptions();
    $this->tempstore = $tempstore_factory->get('sphoenix_ai_auth');
    $this->currentUser = $current_user;
    $this->logger = $logger;
  }

  /**
   * Generate login URL for redirect (like VS Code extension).
   */
  public function getLoginUrl(): string
  {
    $callback_url = Url::fromRoute('sphoenix_ai.auth_callback', [], [
      'absolute' => TRUE,
    ])->toString();

    $current_request = \Drupal::request();
    $current_url = $current_request->getSchemeAndHttpHost() . $current_request->getRequestUri();

    $params = [
      'redirect_uri' => $callback_url,
      'source' => 'drupal_cms',
      'category' => 'cms',
      'destination' => $current_url
    ];

    return Constants::getFrontendBaseUrl() . '/cms-login?' . http_build_query($params);
  }

  /**
   * Generate login URL for redirect (like VS Code extension).
   */
  public function getPlansUrl(): string
  {
    return Constants::getPlansUrl();
  }


  /**
   * Handle OAuth callback with token (like VS Code protocolHandler).
   */
  public function handleAuthCallback(string $token): array
  {
    try {
      // Validate token format (basic JWT check)
      if (!$this->isValidJWTFormat($token)) {
        throw new \Exception('Invalid token format received');
      }

      // Store token temporarily
      $this->storeAccessToken($token);

      // Validate token with server
      if (!$this->validateTokenWithServer($token)) {
        $this->clearAuthData();
        throw new \Exception('Token validation failed');
      }

      // Get user info and subscription (with CMS filtering)
      $user_info = $this->getCurrentUserInfo();

      $subscription_data = $this->fetchSubscriptionData();

      if (!$user_info) {
        $this->clearAuthData();
        throw new \Exception('Failed to get user information');
      }

      // Check if user has CMS subscription
      if (!$subscription_data) {
        // Store auth but indicate no valid subscription
        $auth_data = [
          'access_token' => $token,
          'user_info' => $user_info,
          'subscription' => null,
          'expires_at' => time() + 3600,
          'authenticated_at' => time(),
        ];

        $this->storeAuthData($auth_data);

        return [
          'success' => TRUE,
          'message' => 'Authentication successful but no CMS subscription found',
          'data' => $auth_data,
          'requires_subscription' => TRUE,
        ];
      }

      // Store complete auth data
      $auth_data = [
        'access_token' => $token,
        'user_info' => $user_info,
        'subscription' => $subscription_data,
        'expires_at' => time() + 3600,
        'authenticated_at' => time(),
      ];

      $this->storeAuthData($auth_data);

      $this->logger->info('Authentication successful for user @email with CMS subscription', [
        '@email' => $user_info['email'] ?? 'unknown',
      ]);

      return [
        'success' => TRUE,
        'message' => 'Authentication successful',
        'data' => $auth_data,
      ];
    } catch (\Exception $e) {
      $this->logger->error('Authentication callback failed: @error', [
        '@error' => $e->getMessage(),
      ]);

      return [
        'success' => FALSE,
        'message' => $e->getMessage(),
      ];
    }
  }

  /**
   * Check if user is authenticated.
   */
  public function isAuthenticated(): bool
  {
    $auth_data = $this->getAccess_Token();
    if (!$auth_data) {
      return FALSE;
    }
    return TRUE;
  }

  /**
   * Get current access token.
   */
  public function getAccessToken(): ?string
  {
    if (!$this->isAuthenticated()) {
      return NULL;
    }

    $auth_data = $this->getAccess_Token();
    return $auth_data ?? NULL;
  }

  /**
   * Get authentication status (like VS Code getAuthState).
   */
  public function getAuthState(): array
  {
    $auth_data = $this->getAuthData();
    if (!$auth_data) {
      return [
        'isAuthenticated' => FALSE,
        'user' => NULL,
        'token' => NULL,
        'subscription' => NULL,
      ];
    }

    $is_authenticated = $this->isAuthenticated();

    return [
      'isAuthenticated' => $is_authenticated,
      'user' => $is_authenticated ? ($auth_data['user_info'] ?? NULL) : NULL,
      'token' => $is_authenticated ? $this->getAccessToken() : NULL,
      'subscription' => $is_authenticated ? ($auth_data['subscription'] ?? NULL) : NULL,
    ];
  }

  /**
   * Get current user info.
   */
  protected function getCurrentUserInfo(): ?array
  {
    $token = $this->getAccessToken();
    if (!$token) {
      return NULL;
    }

    try {
      $response = $this->httpClient->request('GET', Constants::getEndpointUrl('USER_PROFILE'), [
        'headers' => [
          'Authorization' => 'Bearer ' . $token,
          'Content-Type' => 'application/json',
        ],
        'timeout' => 30,
      ]);
      $data = json_decode($response->getBody()->getContents(), TRUE);

      if (isset($data['messages'][0]['code']) && $data['messages'][0]['code'] == 200 && isset($data['data']['user_id'])) {
        return $data['data'];
      }

      return NULL;
    } catch (GuzzleException $e) {
      $this->logger->error('Failed to get user info: @error', [
        '@error' => $e->getMessage(),
      ]);
      return NULL;
    }
  }

  /**
   * Fetch subscription data (like VS Code fetchCurrentSubscription).
   */
  protected function fetchSubscriptionData(): ?array
  {
    $token = $this->getAccessToken();
    if (!$token) {
      return NULL;
    }

    try {
      $response = $this->httpClient->request('GET', Constants::getEndpointUrl('SUBSCRIPTION'), [
        'headers' => [
          'Authorization' => 'Bearer ' . $token,
          'Content-Type' => 'application/json',
        ],
        'timeout' => 30,
      ]);

      $data = json_decode($response->getBody()->getContents(), TRUE);
      if (isset($data['messages'][0]['code']) && $data['messages'][0]['code'] == 200 && is_array($data['data'])) {
        $subscriptions = $data['data'];

        // Filter for CMS category subscriptions
        $cms_subscriptions = array_filter($subscriptions, function ($sub) {
          return isset($sub['plan_id']['category']) && $sub['plan_id']['category'] === 'cms';
        });

        if (empty($cms_subscriptions)) {
          return NULL;
        }

        // Convert to format similar to VS Code extension
        return $this->formatSubscriptionData($cms_subscriptions);
      }

      return NULL;
    } catch (GuzzleException $e) {
      $this->logger->error('Failed to get subscription data: @error', [
        '@error' => $e->getMessage(),
      ]);
      return NULL;
    }
  }

  /**
   * Format subscription data like VS Code extension.
   */
  protected function formatSubscriptionData(array $subscriptions): array
  {
    $active_subscriptions = array_filter($subscriptions, function ($sub) {
      return $sub['status'] === 'active';
    });

    if (empty($active_subscriptions)) {
      return [];
    }

    $total_remaining_usage = array_sum(array_column($active_subscriptions, 'remaining_usage'));
    $total_monthly_limit = array_sum(array_map(function ($sub) {
      return $sub['plan_id']['monthly_usage_limit'];
    }, $active_subscriptions));

    // Find highest tier plan
    $highest_tier_sub = array_reduce($active_subscriptions, function ($highest, $current) {
      return ($current['plan_id']['monthly_price'] > $highest['plan_id']['monthly_price']) ? $current : $highest;
    });

    // Primary subscription (most recent)
    $primary_subscription = array_reduce($active_subscriptions, function ($latest, $current) {
      return (strtotime($current['start_date']) > strtotime($latest['start_date'])) ? $current : $latest;
    });

    return [
      'total_remaining_usage' => $total_remaining_usage,
      'active_subscriptions_count' => count($active_subscriptions),
      'highest_tier_plan' => $highest_tier_sub['plan_id'],
      'total_monthly_limit' => $total_monthly_limit,
      'primary_subscription' => $primary_subscription,
      'all_subscriptions' => $active_subscriptions,
    ];
  }

  /**
   * Validate token format (basic JWT check).
   */
  protected function isValidJWTFormat(string $token): bool
  {
    $parts = explode('.', $token);
    return count($parts) === 3 &&
      preg_match('/^[A-Za-z0-9_-]+$/', $parts[0]) &&
      preg_match('/^[A-Za-z0-9_-]+$/', $parts[1]) &&
      preg_match('/^[A-Za-z0-9_-]+$/', $parts[2]);
  }

  /**
   * Validate token with server.
   */
  protected function validateTokenWithServer(string $token): bool
  {
    try {
      $response = $this->httpClient->request('GET', Constants::getEndpointUrl('AUTH_CHECK'), [
        'headers' => [
          'Authorization' => 'Bearer ' . $token,
          'Content-Type' => 'application/json',
        ],
        'timeout' => 30,
      ]);

      return $response->getStatusCode() === 200;
    } catch (GuzzleException $e) {
      return FALSE;
    }
  }

  /**
   * Get subscription data.
   */
  public function getSubscription(): ?array
  {
    if (!$this->isAuthenticated()) {
      return NULL;
    }

    $auth_data = $this->getAuthData();
    return $auth_data['subscription'] ?? NULL;
  }

  /**
   * Decrement usage (like VS Code extension).
   */
  public function decrementUsage(int $amount = 1): void
  {
    if (!$this->isAuthenticated()) {
      return;
    }

    $auth_data = $this->getAuthData();

    if (isset($auth_data['subscription']['total_remaining_usage'])) {
      $auth_data['subscription']['total_remaining_usage'] = max(
        0,
        $auth_data['subscription']['total_remaining_usage'] - $amount
      );
      $this->storeAuthData($auth_data);
    }
  }

  /**
   * Check if user has active subscription.
   */
  public function hasActiveSubscription(): bool
  {
    $subscription = $this->getSubscription();
    return $subscription && $subscription['active_subscriptions_count'] > 0;
  }

  public function clearAuth_Data(): bool
  {
    $this->clearAuthData();
    return true;
  }

  /**
   * Logout user.
   */
  public function logout(): bool
  {
    try {
      $this->clearAuthData();
      return TRUE;
    } catch (\Exception $e) {
      return FALSE;
    }
  }

  /**
   * Store access token.
   */
  protected function storeAccessToken(string $token): void
  {

    $key = 'access_token_' . $this->currentUser->id();
    $this->tempstore->set($key, $token);
  }

  protected function getAccess_Token(): string | null
  {

    $key = 'access_token_' . $this->currentUser->id();
    return $this->tempstore->get($key);
  }

  /**
   * Store authentication data.
   */
  protected function storeAuthData(array $auth_data): void
  {
    $key = 'auth_data_' . $this->currentUser->id();
    $this->tempstore->set($key, $auth_data);
  }

  /**
   * Get authentication data.
   */
  protected function getAuthData(): ?array
  {
    $key = 'auth_data_' . $this->currentUser->id();
    return $this->tempstore->get($key);
  }

  /**
   * Clear authentication data.
   */
  protected function clearAuthData(): void
  {
    $keys = [
      'auth_data_' . $this->currentUser->id(),
      'access_token_' . $this->currentUser->id(),
    ];

    foreach ($keys as $key) {
      $this->tempstore->delete($key);
    }
  }

  public function refreshAuthState(): array
  {
    try {
      // Get existing token from storage
      $existing_token = $this->getAccess_Token();

      if (!$existing_token) {
        return [
          'success' => FALSE,
          'message' => 'No existing token found. Please login again.',
          'data' => [
            'isAuthenticated' => FALSE,
            'user' => NULL,
            'token' => NULL,
            'subscription' => NULL,
          ],
        ];
      }

      // Validate existing token with server
      if (!$this->validateTokenWithServer($existing_token)) {
        $this->clearAuthData();
        return [
          'success' => FALSE,
          'message' => 'Token expired or invalid. Please login again.',
          'data' => [
            'isAuthenticated' => FALSE,
            'user' => NULL,
            'token' => NULL,
            'subscription' => NULL,
          ],
        ];
      }

      // Fetch fresh user info from API
      $user_info = $this->getCurrentUserInfo();
      if (!$user_info) {
        $this->clearAuthData();
        return [
          'success' => FALSE,
          'message' => 'Failed to get user information. Please login again.',
          'data' => [
            'isAuthenticated' => FALSE,
            'user' => NULL,
            'token' => NULL,
            'subscription' => NULL,
          ],
        ];
      }

      // Fetch fresh subscription data from API
      $subscription_data = $this->fetchSubscriptionData();

      // Prepare fresh auth data
      $fresh_auth_data = [
        'access_token' => $existing_token,
        'user_info' => $user_info,
        'subscription' => $subscription_data,
        'expires_at' => time() + 3600,
        'authenticated_at' => time(),
        'refreshed_at' => time(), // Add refresh timestamp
      ];

      // Store the fresh data
      $this->storeAuthData($fresh_auth_data);

      $this->logger->info('Authentication data refreshed for user @email', [
        '@email' => $user_info['email'] ?? 'unknown',
      ]);

      // Return fresh auth state
      $fresh_auth_state = [
        'isAuthenticated' => TRUE,
        'user' => $user_info,
        'token' => $existing_token,
        'subscription' => $subscription_data,
      ];

      if (!$subscription_data) {
        return [
          'success' => TRUE,
          'message' => 'Authentication refreshed but no CMS subscription found',
          'data' => $fresh_auth_state,
          'requires_subscription' => TRUE,
        ];
      }

      return [
        'success' => TRUE,
        'message' => 'Authentication and subscription data refreshed successfully',
        'data' => $fresh_auth_state,
      ];
    } catch (\Exception $e) {
      $this->logger->error('Authentication refresh failed: @error', [
        '@error' => $e->getMessage(),
      ]);

      return [
        'success' => FALSE,
        'message' => $e->getMessage(),
        'data' => [
          'isAuthenticated' => FALSE,
          'user' => NULL,
          'token' => NULL,
          'subscription' => NULL,
        ],
      ];
    }
  }
}
