<?php

namespace Drupal\sphoenix_ai\Service;

use Drupal\sphoenix_ai\Service\FieldMapperService;
use Drupal\Core\Entity\EntityTypeManagerInterface;

/**
 * Service for analyzing content and providing AI suggestions.
 */
class ContentAnalyzerService
{

  /**
   * The field mapper service.
   *
   * @var \Drupal\sphoenix_ai\Service\FieldMapperService
   */
  protected $fieldMapper;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * Constructs a ContentAnalyzerService object.
   *
   * @param \Drupal\sphoenix_ai\Service\FieldMapperService $field_mapper
   *   The field mapper service.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   */
  public function __construct(FieldMapperService $field_mapper, EntityTypeManagerInterface $entity_type_manager)
  {
    $this->fieldMapper = $field_mapper;
    $this->entityTypeManager = $entity_type_manager;
  }

  /**
   * Analyze content fields and suggest AI tasks.
   *
   * @param string $content_type
   *   The content type machine name.
   * @param array $field_data
   *   The current field data.
   *
   * @return array
   *   Array of AI task suggestions.
   */
  public function analyzeAndSuggest($content_type, array $field_data)
  {
    $suggestions = [];
    $text_fields = $this->fieldMapper->getTextFields($content_type);
    $extracted_content = $this->fieldMapper->extractExistingContent($content_type, $field_data);

    // Analyze each field and suggest appropriate AI tasks.
    foreach ($text_fields as $field_name => $field_info) {
      $field_suggestions = $this->analyzeField($field_name, $field_info, $extracted_content);
      $suggestions = array_merge($suggestions, $field_suggestions);
    }

    // Add global suggestions based on overall content state.
    $global_suggestions = $this->analyzeGlobalContent($content_type, $extracted_content);
    $suggestions = array_merge($suggestions, $global_suggestions);

    // Sort suggestions by priority.
    usort($suggestions, function ($a, $b) {
      $priority_order = ['high' => 3, 'medium' => 2, 'low' => 1];
      return $priority_order[$b['priority']] <=> $priority_order[$a['priority']];
    });

    return array_slice($suggestions, 0, 10); // Limit to top 10 suggestions.
  }

  /**
   * Analyze a specific field and suggest AI tasks.
   *
   * @param string $field_name
   *   The field name.
   * @param array $field_info
   *   The field information.
   * @param array $extracted_content
   *   The extracted content from all fields.
   *
   * @return array
   *   Array of suggestions for this field.
   */
  protected function analyzeField($field_name, array $field_info, array $extracted_content)
  {
    $suggestions = [];
    $field_content = $extracted_content[$field_name] ?? null;

    // Field is empty.
    if (!$field_content) {
      $task_type = $this->suggestTaskForEmptyField($field_name, $field_info);
      if ($task_type) {
        $suggestions[] = [
          'task_type' => $task_type,
          'field_name' => $field_name,
          'field_label' => $field_info['label'],
          'priority' => $field_info['required'] ? 'high' : 'medium',
          'message' => $this->generateSuggestionMessage($task_type, $field_info['label'], 'empty'),
          'icon' => $this->getTaskIcon($task_type),
          'action' => 'generate',
        ];
      }
    }
    // Field has content but might need improvement.
    else {
      $content_length = $field_content['length'];
      $content_value = $field_content['value'];

      // Content too short.
      if ($content_length < $this->getMinimumLength($field_name)) {
        $suggestions[] = [
          'task_type' => 'content_regeneration',
          'field_name' => $field_name,
          'field_label' => $field_info['label'],
          'priority' => 'medium',
          'message' => $this->t('Expand content in @field', ['@field' => $field_info['label']]),
          'icon' => '📝',
          'action' => 'expand',
          'current_length' => $content_length,
          'suggested_length' => $this->getRecommendedLength($field_name),
        ];
      }

      // Content too long.
      if ($content_length > $this->getMaximumLength($field_name)) {
        $suggestions[] = [
          'task_type' => 'content_summary',
          'field_name' => $field_name,
          'field_label' => $field_info['label'],
          'priority' => 'low',
          'message' => $this->t('Summarize content in @field', ['@field' => $field_info['label']]),
          'icon' => '📄',
          'action' => 'summarize',
          'current_length' => $content_length,
          'suggested_length' => $this->getRecommendedLength($field_name),
        ];
      }

      // Analyze content quality.
      $quality_issues = $this->analyzeContentQuality($content_value, $field_name);
      foreach ($quality_issues as $issue) {
        $suggestions[] = array_merge($issue, [
          'field_name' => $field_name,
          'field_label' => $field_info['label'],
        ]);
      }
    }

    return $suggestions;
  }

  /**
   * Analyze global content state and suggest AI tasks.
   *
   * @param string $content_type
   *   The content type.
   * @param array $extracted_content
   *   The extracted content from all fields.
   *
   * @return array
   *   Array of global suggestions.
   */
  protected function analyzeGlobalContent($content_type, array $extracted_content)
  {
    $suggestions = [];
    $total_content_length = array_sum(array_column($extracted_content, 'length'));

    // No content at all.
    if (empty($extracted_content)) {
      $suggestions[] = [
        'task_type' => 'content_generation',
        'field_name' => null,
        'field_label' => null,
        'priority' => 'high',
        'message' => $this->t('Generate initial content for this @type', ['@type' => $content_type]),
        'icon' => '✍️',
        'action' => 'create',
      ];
    }
    // Has some content but might benefit from optimization.
    else {
      // Suggest SEO analysis if content exists.
      if ($total_content_length > 100) {
        $suggestions[] = [
          'task_type' => 'seo_analysis',
          'field_name' => null,
          'field_label' => null,
          'priority' => 'medium',
          'message' => $this->t('Analyze content for SEO optimization'),
          'icon' => '🔍',
          'action' => 'analyze',
        ];

        $suggestions[] = [
          'task_type' => 'content_analysis',
          'field_name' => null,
          'field_label' => null,
          'priority' => 'medium',
          'message' => $this->t('Analyze content quality and readability'),
          'icon' => '📊',
          'action' => 'analyze',
        ];
      }

      // Suggest keyword extraction.
      if ($total_content_length > 200) {
        $suggestions[] = [
          'task_type' => 'keyword_extraction',
          'field_name' => null,
          'field_label' => null,
          'priority' => 'low',
          'message' => $this->t('Extract keywords from existing content'),
          'icon' => '🔖',
          'action' => 'extract',
        ];
      }

      // Suggest meta generation if no meta fields are filled.
      $has_meta = $this->hasMetaContent($extracted_content);
      if (!$has_meta && $total_content_length > 50) {
        $suggestions[] = [
          'task_type' => 'meta_generation',
          'field_name' => null,
          'field_label' => null,
          'priority' => 'high',
          'message' => $this->t('Generate SEO meta tags and descriptions'),
          'icon' => '🏷️',
          'action' => 'generate',
        ];
      }

      // Suggest thumbnail if no image content.
      if (!$this->hasImageContent($extracted_content)) {
        $suggestions[] = [
          'task_type' => 'thumbnail_generation',
          'field_name' => null,
          'field_label' => null,
          'priority' => 'medium',
          'message' => $this->t('Generate thumbnail image for content'),
          'icon' => '🖼️',
          'action' => 'generate',
        ];
      }
    }

    return $suggestions;
  }

  /**
   * Suggest appropriate AI task for an empty field.
   *
   * @param string $field_name
   *   The field name.
   * @param array $field_info
   *   The field information.
   *
   * @return string|null
   *   The suggested task type or NULL.
   */
  protected function suggestTaskForEmptyField($field_name, array $field_info)
  {
    // Title fields.
    if (strpos($field_name, 'title') !== false) {
      return 'title_generation';
    }

    // Body/content fields.
    if (in_array($field_name, ['body', 'field_body', 'field_content'])) {
      return 'content_generation';
    }

    // Summary/description fields.
    if (strpos($field_name, 'summary') !== false || strpos($field_name, 'description') !== false) {
      return 'description_generation';
    }

    // Meta fields.
    if (strpos($field_name, 'meta') !== false) {
      return 'meta_generation';
    }

    // For other text fields, suggest content generation.
    if (in_array($field_info['type'], ['text_long', 'text_with_summary'])) {
      return 'content_generation';
    }

    // For short text fields, suggest description generation.
    if (in_array($field_info['type'], ['string', 'string_long', 'text'])) {
      return 'description_generation';
    }

    return null;
  }

  /**
   * Analyze content quality and identify issues.
   *
   * @param string $content
   *   The content to analyze.
   * @param string $field_name
   *   The field name.
   *
   * @return array
   *   Array of quality issues found.
   */
  protected function analyzeContentQuality($content, $field_name)
  {
    $issues = [];

    // Check for repetitive content.
    if ($this->hasRepetitiveContent($content)) {
      $issues[] = [
        'task_type' => 'content_optimization',
        'priority' => 'medium',
        'message' => $this->t('Content appears repetitive - optimize for variety'),
        'icon' => '🔄',
        'action' => 'optimize',
        'issue_type' => 'repetitive',
      ];
    }

    // Check for poor readability.
    if ($this->hasPoorReadability($content)) {
      $issues[] = [
        'task_type' => 'content_optimization',
        'priority' => 'medium',
        'message' => $this->t('Improve content readability and structure'),
        'icon' => '📖',
        'action' => 'optimize',
        'issue_type' => 'readability',
      ];
    }

    // Check for lack of structure in long content.
    if (strlen($content) > 1000 && !$this->hasGoodStructure($content)) {
      $issues[] = [
        'task_type' => 'content_optimization',
        'priority' => 'low',
        'message' => $this->t('Add headings and structure to long content'),
        'icon' => '📋',
        'action' => 'structure',
        'issue_type' => 'structure',
      ];
    }

    return $issues;
  }

  /**
   * Check if content has repetitive patterns.
   *
   * @param string $content
   *   The content to check.
   *
   * @return bool
   *   TRUE if content appears repetitive.
   */
  protected function hasRepetitiveContent($content)
  {
    $sentences = preg_split('/[.!?]+/', $content);
    $sentences = array_filter(array_map('trim', $sentences));

    if (count($sentences) < 3) {
      return FALSE;
    }

    // Check for similar sentence starts.
    $starts = [];
    foreach ($sentences as $sentence) {
      $words = explode(' ', $sentence);
      if (count($words) >= 3) {
        $start = implode(' ', array_slice($words, 0, 3));
        $starts[] = strtolower($start);
      }
    }

    $start_counts = array_count_values($starts);
    $max_count = max($start_counts);

    return $max_count > (count($sentences) * 0.3); // More than 30% similar starts.
  }

  /**
   * Check if content has poor readability.
   *
   * @param string $content
   *   The content to check.
   *
   * @return bool
   *   TRUE if readability appears poor.
   */
  protected function hasPoorReadability($content)
  {
    $sentences = preg_split('/[.!?]+/', $content);
    $sentences = array_filter(array_map('trim', $sentences));

    if (empty($sentences)) {
      return TRUE;
    }

    $total_words = 0;
    $long_sentences = 0;

    foreach ($sentences as $sentence) {
      $word_count = str_word_count($sentence);
      $total_words += $word_count;

      if ($word_count > 25) { // Very long sentences.
        $long_sentences++;
      }
    }

    $avg_words_per_sentence = $total_words / count($sentences);
    $long_sentence_ratio = $long_sentences / count($sentences);

    // Poor readability indicators.
    return $avg_words_per_sentence > 20 || $long_sentence_ratio > 0.4;
  }

  /**
   * Check if content has good structure.
   *
   * @param string $content
   *   The content to check.
   *
   * @return bool
   *   TRUE if content has good structure.
   */
  protected function hasGoodStructure($content)
  {
    // Check for headings.
    $has_headings = preg_match('/<h[1-6]/', $content) ||
      preg_match('/^#+\s/m', $content) || // Markdown headings.
      preg_match('/\n\n[A-Z][^.]*:?\n/m', $content); // Simple headings.

    // Check for lists.
    $has_lists = preg_match('/<[uo]l>/', $content) ||
      preg_match('/^\s*[-*+]\s/m', $content) || // Markdown lists.
      preg_match('/^\s*\d+\.\s/m', $content); // Numbered lists.

    // Check for paragraphs.
    $paragraph_count = substr_count($content, "\n\n") + 1;
    $has_paragraphs = $paragraph_count > 2;

    return $has_headings && ($has_lists || $has_paragraphs);
  }

  /**
   * Check if extracted content includes meta information.
   *
   * @param array $extracted_content
   *   The extracted content.
   *
   * @return bool
   *   TRUE if meta content exists.
   */
  protected function hasMetaContent(array $extracted_content)
  {
    $meta_fields = ['field_meta_title', 'field_meta_description', 'field_seo_title', 'field_seo_description'];

    foreach ($meta_fields as $field) {
      if (isset($extracted_content[$field]) && !empty($extracted_content[$field]['value'])) {
        return TRUE;
      }
    }

    return FALSE;
  }

  /**
   * Check if extracted content includes image information.
   *
   * @param array $extracted_content
   *   The extracted content.
   *
   * @return bool
   *   TRUE if image content exists.
   */
  protected function hasImageContent(array $extracted_content)
  {
    $image_fields = ['field_image', 'field_featured_image', 'field_thumbnail'];

    foreach ($image_fields as $field) {
      if (isset($extracted_content[$field]) && !empty($extracted_content[$field]['value'])) {
        return TRUE;
      }
    }

    return FALSE;
  }

  /**
   * Get minimum content length for a field.
   *
   * @param string $field_name
   *   The field name.
   *
   * @return int
   *   Minimum length in characters.
   */
  protected function getMinimumLength($field_name)
  {
    $minimums = [
      'title' => 10,
      'field_title' => 10,
      'body' => 100,
      'field_body' => 100,
      'field_content' => 100,
      'field_summary' => 50,
      'field_description' => 50,
      'field_meta_title' => 20,
      'field_meta_description' => 80,
    ];

    return $minimums[$field_name] ?? 50;
  }

  /**
   * Get maximum content length for a field.
   *
   * @param string $field_name
   *   The field name.
   *
   * @return int
   *   Maximum length in characters.
   */
  protected function getMaximumLength($field_name)
  {
    $maximums = [
      'title' => 100,
      'field_title' => 100,
      'field_meta_title' => 60,
      'field_meta_description' => 160,
      'field_summary' => 500,
      'field_description' => 500,
    ];

    return $maximums[$field_name] ?? PHP_INT_MAX;
  }

  /**
   * Get recommended content length for a field.
   *
   * @param string $field_name
   *   The field name.
   *
   * @return int
   *   Recommended length in characters.
   */
  protected function getRecommendedLength($field_name)
  {
    $recommendations = [
      'title' => 50,
      'field_title' => 50,
      'body' => 600,
      'field_body' => 600,
      'field_content' => 600,
      'field_summary' => 150,
      'field_description' => 150,
      'field_meta_title' => 50,
      'field_meta_description' => 120,
    ];

    return $recommendations[$field_name] ?? 300;
  }

  /**
   * Generate a suggestion message.
   *
   * @param string $task_type
   *   The AI task type.
   * @param string $field_label
   *   The field label.
   * @param string $context
   *   The context (empty, short, long, etc.).
   *
   * @return string
   *   The suggestion message.
   */
  protected function generateSuggestionMessage($task_type, $field_label, $context)
  {
    $messages = [
      'content_generation' => [
        'empty' => $this->t('Generate content for @field', ['@field' => $field_label]),
        'short' => $this->t('Expand content in @field', ['@field' => $field_label]),
      ],
      'title_generation' => [
        'empty' => $this->t('Generate title for @field', ['@field' => $field_label]),
      ],
      'description_generation' => [
        'empty' => $this->t('Generate description for @field', ['@field' => $field_label]),
      ],
      'meta_generation' => [
        'empty' => $this->t('Generate meta tags for @field', ['@field' => $field_label]),
      ],
    ];

    return $messages[$task_type][$context] ?? $this->t('Improve @field with AI', ['@field' => $field_label]);
  }

  /**
   * Get icon for an AI task type.
   *
   * @param string $task_type
   *   The task type.
   *
   * @return string
   *   The icon character.
   */
  protected function getTaskIcon($task_type)
  {
    $icons = [
      'content_generation' => '✍️',
      'content_regeneration' => '🔄',
      'content_optimization' => '📈',
      'thumbnail_generation' => '🖼️',
      'graph_generation' => '📊',
      'table_generation' => '📋',
      'meta_generation' => '🏷️',
      'seo_analysis' => '🔍',
      'content_analysis' => '📝',
      'keyword_extraction' => '🔖',
      'content_summary' => '📄',
      'title_generation' => '📰',
      'description_generation' => '📖',
    ];

    return $icons[$task_type] ?? '🤖';
  }

  /**
   * Translates a string.
   *
   * @param string $string
   *   The string to translate.
   * @param array $args
   *   Arguments for translation.
   *
   * @return \Drupal\Core\StringTranslation\TranslatableMarkup
   *   The translated string.
   */
  protected function t($string, array $args = [])
  {
    return \Drupal::translation()->translate($string, $args);
  }
}
