<?php

namespace Drupal\sphoenix_ai\Service;

use Drupal\Core\Session\AccountProxyInterface;
use Drupal\sphoenix_ai\Service\ApiClientService;
use Drupal\sphoenix_ai\Service\AuthenticationService;

/**
 * Service for managing SPhoenix AI subscriptions.
 */
class SubscriptionService
{

  /**
   * The API client service.
   *
   * @var \Drupal\sphoenix_ai\Service\ApiClientService
   */
  protected $apiClient;

  /**
   * The authentication service.
   *
   * @var \Drupal\sphoenix_ai\Service\AuthenticationService
   */
  protected $authService;

  /**
   * The current user.
   *
   * @var \Drupal\Core\Session\AccountProxyInterface
   */
  protected $currentUser;

  /**
   * Available subscription plans.
   *
   * @var array
   */
  protected $availablePlans = [
    'free' => [
      'id' => 'free',
      'name' => 'Free',
      'display_name' => 'Free',
      'price' => 0,
      'currency' => 'USD',
      'billing' => 'monthly',
      'description' => 'Perfect for getting started with AI-powered tools',
      'monthly_limit' => 50,
      'features' => [
        '50 requests per month',
        'Basic AI tools',
        'Standard models',
        'Community support',
      ],
    ],
    'starter' => [
      'id' => 'starter',
      'name' => 'starter',
      'display_name' => 'Starter Plan',
      'price' => 20,
      'currency' => 'USD',
      'billing' => 'monthly',
      'description' => 'Affordable access for individual creators',
      'monthly_limit' => 200,
      'popular' => TRUE,
      'features' => [
        '200 requests per month',
        'Full suite of 100+ AI tools',
        'SEO, Content Analysis tools',
        'Agentic Content & Image Generation',
        'Basic exportable reports',
        'Email support',
        'On-demand credit packs',
      ],
    ],
    'pro' => [
      'id' => 'pro',
      'name' => 'pro',
      'display_name' => 'Pro Plan',
      'price' => 50,
      'currency' => 'USD',
      'billing' => 'monthly',
      'description' => 'For professional developers and content creators',
      'monthly_limit' => 1000,
      'features' => [
        '1000 requests per month',
        'All Starter features',
        'Priority processing',
        'Advanced analytics',
        'Priority support',
        'Custom integrations',
      ],
    ],
    'enterprise' => [
      'id' => 'enterprise',
      'name' => 'enterprise',
      'display_name' => 'Enterprise',
      'price' => 200,
      'currency' => 'USD',
      'billing' => 'monthly',
      'description' => 'For teams and organizations with enterprise needs',
      'monthly_limit' => -1, // Unlimited
      'features' => [
        'Unlimited requests',
        'All Pro features',
        'Custom AI models',
        'Team collaboration',
        'Advanced security',
        'Dedicated support',
        'On-premise deployment',
        'SLA guarantee',
      ],
    ],
  ];

  /**
   * Constructs a SubscriptionService object.
   *
   * @param \Drupal\sphoenix_ai\Service\ApiClientService $api_client
   *   The API client service.
   * @param \Drupal\sphoenix_ai\Service\AuthenticationService $auth_service
   *   The authentication service.
   * @param \Drupal\Core\Session\AccountProxyInterface $current_user
   *   The current user.
   */
  public function __construct(ApiClientService $api_client, AuthenticationService $auth_service, AccountProxyInterface $current_user)
  {
    $this->apiClient = $api_client;
    $this->authService = $auth_service;
    $this->currentUser = $current_user;
  }

  /**
   * Get available subscription plans.
   *
   * @return array
   *   Array of available plans.
   */
  public function getAvailablePlans()
  {
    return $this->availablePlans;
  }

  /**
   * Get current subscription information.
   *
   * @return array|null
   *   Current subscription data or NULL if not subscribed.
   */
  public function getCurrentSubscription()
  {
    return $this->authService->getSubscription();
  }

  /**
   * Get usage information for current user.
   *
   * @return array|null
   *   Usage information or NULL if not authenticated.
   */
  public function getUsageInfo()
  {
    if (!$this->authService->isAuthenticated()) {
      return NULL;
    }

    $subscription = $this->getCurrentSubscription();
    if (!$subscription) {
      return NULL;
    }

    $monthly_limit = $subscription['total_monthly_limit'] ?? 0;
    $remaining_usage = $subscription['total_remaining_usage'] ?? 0;
    $used_this_month = max(0, $monthly_limit - $remaining_usage);
    $usage_percentage = $monthly_limit > 0 ? ($used_this_month / $monthly_limit) * 100 : 0;

    return [
      'subscription' => $subscription,
      'used_this_month' => $used_this_month,
      'remaining_usage' => $remaining_usage,
      'monthly_limit' => $monthly_limit,
      'usage_percentage' => $usage_percentage,
      'plan_name' => $subscription['highest_tier_plan']['plan_name'] ?? 'Unknown',
      'status' => $subscription['active_subscriptions_count'] > 0 ? 'active' : 'inactive',
    ];
  }

  /**
   * Check if user can use a feature.
   *
   * @param string $feature
   *   The feature name.
   *
   * @return bool
   *   TRUE if user can use the feature.
   */
  public function canUseFeature($feature)
  {
    if (!$this->authService->isAuthenticated()) {
      return FALSE;
    }

    $subscription = $this->getCurrentSubscription();
    if (!$subscription) {
      return FALSE;
    }

    // Check if any subscription is active
    $has_active_subscription = FALSE;
    if (isset($subscription['all_subscriptions'])) {
      foreach ($subscription['all_subscriptions'] as $sub) {
        if ($sub['status'] === 'active') {
          $has_active_subscription = TRUE;
          break;
        }
      }
    }

    if (!$has_active_subscription) {
      return FALSE;
    }

    // Check remaining usage
    return $subscription['total_remaining_usage'] > 0;
  }

  /**
   * Check if there's remaining usage.
   *
   * @return bool
   *   TRUE if there's remaining usage.
   */
  public function hasUsageRemaining()
  {
    if (!$this->authService->isAuthenticated()) {
      return FALSE;
    }

    $subscription = $this->getCurrentSubscription();
    if (!$subscription) {
      return FALSE;
    }

    $has_active_subscription = FALSE;
    if (isset($subscription['all_subscriptions'])) {
      foreach ($subscription['all_subscriptions'] as $sub) {
        if ($sub['status'] === 'active') {
          $has_active_subscription = TRUE;
          break;
        }
      }
    }

    return $has_active_subscription && $subscription['total_remaining_usage'] > 0;
  }

  /**
   * Record usage for a feature.
   *
   * @param string $feature
   *   The feature name.
   * @param int $amount
   *   The usage amount.
   */
  public function recordUsage($feature, $amount = 1)
  {
    if (!$this->authService->isAuthenticated()) {
      return;
    }

    // Update local subscription data
    $this->authService->decrementUsage($amount);

    // Log usage warning if getting low
    $subscription = $this->getCurrentSubscription();
    if ($subscription) {
      $usage_percentage = $this->calculateUsagePercentage($subscription);

      if ($usage_percentage >= 90) {
        \Drupal::messenger()->addWarning(t('You have used @percentage% of your monthly AI usage limit.', [
          '@percentage' => round($usage_percentage),
        ]));
      }
    }
  }

  /**
   * Get plans page URL.
   *
   * @return string
   *   The plans page URL.
   */
  public function getPlansUrl()
  {
    return $this->authService->getPlansUrl();
  }

  /**
   * Show upgrade prompt message.
   *
   * @param string $feature
   *   The feature name.
   * @param string $reason
   *   Optional reason for the prompt.
   */
  public function showUpgradePrompt($feature, $reason = NULL)
  {
    $subscription = $this->getCurrentSubscription();
    $current_plan = $subscription['highest_tier_plan']['plan_name'] ?? 'No Plan';

    $message = t('Your @plan plan has limited access to @feature.', [
      '@plan' => $current_plan,
      '@feature' => $feature,
    ]);

    if ($reason) {
      $message .= ' ' . $reason;
    }

    if ($subscription && $subscription['total_remaining_usage'] <= 0) {
      $message = t('You\'ve used all @limit requests for this month across your plans.', [
        '@limit' => $subscription['total_monthly_limit'],
      ]);
    }

    $message .= ' ' . t('Would you like to upgrade or add more plans?');

    \Drupal::messenger()->addWarning($message);
  }

  /**
   * Show usage warning based on percentage.
   *
   * @param float $percentage
   *   Usage percentage.
   */
  public function showUsageWarning($percentage)
  {
    if ($percentage >= 90) {
      \Drupal::messenger()->addWarning(t('⚠️ You have used @percentage% of your monthly usage limit.', [
        '@percentage' => round($percentage),
      ]));
    } elseif ($percentage >= 80) {
      \Drupal::messenger()->addMessage(t('You have used @percentage% of your monthly usage limit.', [
        '@percentage' => round($percentage),
      ]));
    }
  }

  /**
   * Get usage recommendations.
   *
   * @return array
   *   Array of usage recommendations.
   */
  public function getUsageRecommendations()
  {
    $recommendations = [];
    $subscription = $this->getCurrentSubscription();

    if (!$subscription) {
      return [t('Login to view usage recommendations')];
    }

    $usage_info = $this->getUsageInfo();
    if (!$usage_info) {
      return [t('Unable to analyze usage patterns')];
    }

    // Usage-based recommendations
    if ($usage_info['usage_percentage'] > 80) {
      if ($subscription['active_subscriptions_count'] > 1) {
        $recommendations[] = t('Consider adding more subscriptions or upgrading your @plan', [
          '@plan' => $subscription['highest_tier_plan']['plan_name'],
        ]);
      } else {
        $recommendations[] = t('Consider upgrading from @plan for higher limits', [
          '@plan' => $subscription['highest_tier_plan']['plan_name'],
        ]);
      }
    }

    if ($subscription['total_remaining_usage'] < 10 && $subscription['total_monthly_limit'] > 0) {
      $recommendations[] = t('You\'re running low on requests - consider adding more plans before month end');
    }

    $avg_plan_price = 0;
    if (isset($subscription['all_subscriptions']) && count($subscription['all_subscriptions']) > 0) {
      $total_price = 0;
      foreach ($subscription['all_subscriptions'] as $sub) {
        $total_price += $sub['plan_id']['monthly_price'] ?? 0;
      }
      $avg_plan_price = $total_price / count($subscription['all_subscriptions']);
    }

    if ($avg_plan_price <= 30 && $usage_info['used_this_month'] > 150) {
      $recommendations[] = t('You\'re an active user! Higher tier plans or additional subscriptions offer better value');
    }

    if ($subscription['active_subscriptions_count'] > 1) {
      $recommendations[] = t('Great strategy having @count active plans for maximum flexibility!', [
        '@count' => $subscription['active_subscriptions_count'],
      ]);
    }

    if (empty($recommendations)) {
      $recommendations[] = t('You\'re managing your usage well! Keep up the great work.');
    }

    return $recommendations;
  }

  /**
   * Calculate usage percentage from subscription data.
   *
   * @param array $subscription
   *   Subscription data.
   *
   * @return float
   *   Usage percentage.
   */
  protected function calculateUsagePercentage(array $subscription)
  {
    $monthly_limit = $subscription['total_monthly_limit'] ?? 0;
    $remaining_usage = $subscription['total_remaining_usage'] ?? 0;

    if ($monthly_limit <= 0) {
      return 0;
    }

    $used_this_month = max(0, $monthly_limit - $remaining_usage);
    return ($used_this_month / $monthly_limit) * 100;
  }

  /**
   * Get plan by ID.
   *
   * @param string $plan_id
   *   The plan ID.
   *
   * @return array|null
   *   Plan data or NULL if not found.
   */
  public function getPlanById($plan_id)
  {
    return $this->availablePlans[$plan_id] ?? NULL;
  }

  /**
   * Check if user has active subscription.
   *
   * @return bool
   *   TRUE if user has active subscription.
   */
  public function hasActiveSubscription()
  {
    return $this->authService->hasActiveSubscription();
  }

  /**
   * Get current plan name.
   *
   * @return string
   *   Current plan name or 'Free' if no subscription.
   */
  public function getCurrentPlanName()
  {
    $subscription = $this->getCurrentSubscription();

    if (!$subscription) {
      return 'Free';
    }

    return $subscription['highest_tier_plan']['plan_name'] ?? 'Unknown';
  }

  /**
   * Format subscription status for display.
   *
   * @return array
   *   Formatted subscription status.
   */
  public function formatSubscriptionStatus()
  {
    if (!$this->authService->isAuthenticated()) {
      return [
        'status' => 'not_authenticated',
        'message' => 'Please login to access AI features',
        'action_url' => NULL,
        'action_text' => 'Login',
      ];
    }

    $subscription = $this->getCurrentSubscription();

    if (!$subscription || $subscription['active_subscriptions_count'] == 0) {
      return [
        'status' => 'no_subscription',
        'message' => 'No active subscription found',
        'action_url' => $this->getPlansUrl(),
        'action_text' => 'View Plans',
      ];
    }

    $usage_info = $this->getUsageInfo();

    if ($usage_info['remaining_usage'] <= 0) {
      return [
        'status' => 'limit_exceeded',
        'message' => 'Monthly usage limit exceeded',
        'action_url' => $this->getPlansUrl(),
        'action_text' => 'Upgrade Plan',
      ];
    }

    return [
      'status' => 'active',
      'message' => $usage_info['remaining_usage'] . ' requests remaining this month',
      'action_url' => NULL,
      'action_text' => NULL,
    ];
  }
}
