<?php

namespace Drupal\sphoenix_ai\Service;

use Drupal\Core\Database\Connection;
use Drupal\Core\Session\AccountProxyInterface;
use Drupal\Component\Datetime\TimeInterface;
use Drupal\node\NodeInterface;

/**
 * Service for tracking AI Content Assistant usage.
 */
class UsageTrackerService
{

  /**
   * The database connection.
   *
   * @var \Drupal\Core\Database\Connection
   */
  protected $database;

  /**
   * The current user.
   *
   * @var \Drupal\Core\Session\AccountProxyInterface
   */
  protected $currentUser;

  /**
   * The time service.
   *
   * @var \Drupal\Component\Datetime\TimeInterface
   */
  protected $time;

  /**
   * Constructs a UsageTrackerService object.
   *
   * @param \Drupal\Core\Database\Connection $database
   *   The database connection.
   * @param \Drupal\Core\Session\AccountProxyInterface $current_user
   *   The current user.
   * @param \Drupal\Component\Datetime\TimeInterface $time
   *   The time service.
   */
  public function __construct(Connection $database, AccountProxyInterface $current_user, TimeInterface $time)
  {
    $this->database = $database;
    $this->currentUser = $current_user;
    $this->time = $time;
  }

  /**
   * Track an AI request.
   *
   * @param string $task_type
   *   The type of AI task performed.
   * @param array $metadata
   *   Additional metadata about the request.
   *
   * @return int
   *   The ID of the usage record.
   */
  public function trackRequest($task_type, array $metadata = [])
  {
    $record = [
      'uid' => $this->currentUser->id(),
      'task_type' => $task_type,
      'status' => 'requested',
      'metadata' => json_encode($metadata),
      'created' => $this->time->getRequestTime(),
      'updated' => $this->time->getRequestTime(),
    ];

    return $this->database->insert('sphoenix_ai_usage')
      ->fields($record)
      ->execute();
  }

  /**
   * Track a completed AI request.
   *
   * @param string $task_type
   *   The type of AI task performed.
   * @param array $response_data
   *   The AI response data.
   * @param int $usage_id
   *   The usage record ID to update (optional).
   *
   * @return int
   *   The ID of the usage record.
   */
  public function trackCompletion($task_type, array $response_data, $usage_id = NULL)
  {
    $metadata = [
      'tokens_used' => $response_data['tokens_used'] ?? 0,
      'cost_estimate' => $response_data['cost_estimate'] ?? 0,
      'processing_time' => $response_data['processing_time'] ?? 0,
      'humanization_applied' => $response_data['humanization_applied'] ?? FALSE,
      'response_length' => $this->calculateResponseLength($response_data),
    ];

    if ($usage_id) {
      // Update existing record.
      $this->database->update('sphoenix_ai_usage')
        ->fields([
          'status' => 'completed',
          'metadata' => json_encode($metadata),
          'updated' => $this->time->getRequestTime(),
        ])
        ->condition('id', $usage_id)
        ->execute();

      return $usage_id;
    } else {
      // Create new record.
      $record = [
        'uid' => $this->currentUser->id(),
        'task_type' => $task_type,
        'status' => 'completed',
        'metadata' => json_encode($metadata),
        'created' => $this->time->getRequestTime(),
        'updated' => $this->time->getRequestTime(),
      ];

      return $this->database->insert('sphoenix_ai_usage')
        ->fields($record)
        ->execute();
    }
  }

  /**
   * Track an error in AI request.
   *
   * @param string $task_type
   *   The type of AI task that failed.
   * @param string $error_message
   *   The error message.
   * @param int $usage_id
   *   The usage record ID to update (optional).
   *
   * @return int
   *   The ID of the usage record.
   */
  public function trackError($task_type, $error_message, $usage_id = NULL)
  {
    $metadata = [
      'error_message' => $error_message,
      'error_time' => $this->time->getRequestTime(),
    ];

    if ($usage_id) {
      // Update existing record.
      $this->database->update('sphoenix_ai_usage')
        ->fields([
          'status' => 'error',
          'metadata' => json_encode($metadata),
          'updated' => $this->time->getRequestTime(),
        ])
        ->condition('id', $usage_id)
        ->execute();

      return $usage_id;
    } else {
      // Create new record.
      $record = [
        'uid' => $this->currentUser->id(),
        'task_type' => $task_type,
        'status' => 'error',
        'metadata' => json_encode($metadata),
        'created' => $this->time->getRequestTime(),
        'updated' => $this->time->getRequestTime(),
      ];

      return $this->database->insert('sphoenix_ai_usage')
        ->fields($record)
        ->execute();
    }
  }

  /**
   * Track when AI-assisted content is saved.
   *
   * @param \Drupal\node\NodeInterface $node
   *   The node being saved.
   * @param array $tasks_used
   *   List of AI tasks used for this content.
   */
  public function trackContentSave(NodeInterface $node, array $tasks_used = [])
  {
    $metadata = [
      'node_id' => $node->id(),
      'content_type' => $node->bundle(),
      'node_title' => $node->getTitle(),
      'tasks_used' => $tasks_used,
      'is_new' => $node->isNew(),
    ];

    $record = [
      'uid' => $this->currentUser->id(),
      'task_type' => 'content_save',
      'status' => 'completed',
      'metadata' => json_encode($metadata),
      'created' => $this->time->getRequestTime(),
      'updated' => $this->time->getRequestTime(),
    ];

    $this->database->insert('sphoenix_ai_usage')
      ->fields($record)
      ->execute();
  }

  /**
   * Get usage statistics for a user.
   *
   * @param int $uid
   *   The user ID.
   * @param int $days
   *   Number of days to look back.
   *
   * @return array
   *   Usage statistics.
   */
  public function getUserStats($uid, $days = 30)
  {
    $start_time = $this->time->getRequestTime() - ($days * 24 * 60 * 60);

    // Total requests.
    $total_requests = $this->database->query(
      "SELECT COUNT(*) FROM {sphoenix_ai_usage} WHERE uid = :uid AND created >= :start_time",
      [':uid' => $uid, ':start_time' => $start_time]
    )->fetchField();

    // Requests by task type.
    $task_stats = $this->database->query(
      "SELECT task_type, COUNT(*) as count, AVG(JSON_EXTRACT(metadata, '$.processing_time')) as avg_time
       FROM {sphoenix_ai_usage}
       WHERE uid = :uid AND created >= :start_time AND status = 'completed'
       GROUP BY task_type
       ORDER BY count DESC",
      [':uid' => $uid, ':start_time' => $start_time]
    )->fetchAllAssoc('task_type');

    // Requests by status.
    $status_stats = $this->database->query(
      "SELECT status, COUNT(*) as count
       FROM {sphoenix_ai_usage}
       WHERE uid = :uid AND created >= :start_time
       GROUP BY status",
      [':uid' => $uid, ':start_time' => $start_time]
    )->fetchAllKeyed();

    // Daily usage for the last 7 days.
    $daily_usage = [];
    for ($i = 6; $i >= 0; $i--) {
      $day_start = strtotime("-{$i} days", $this->time->getRequestTime());
      $day_start = mktime(0, 0, 0, date('m', $day_start), date('d', $day_start), date('Y', $day_start));
      $day_end = $day_start + (24 * 60 * 60);

      $count = $this->database->query(
        "SELECT COUNT(*) FROM {sphoenix_ai_usage} WHERE uid = :uid AND created >= :start AND created < :end",
        [':uid' => $uid, ':start' => $day_start, ':end' => $day_end]
      )->fetchField();

      $daily_usage[date('Y-m-d', $day_start)] = (int) $count;
    }

    // Calculate totals.
    $total_tokens = 0;
    $total_cost = 0;
    $total_time = 0;

    $cost_query = $this->database->query(
      "SELECT metadata FROM {sphoenix_ai_usage} WHERE uid = :uid AND created >= :start_time AND status = 'completed'",
      [':uid' => $uid, ':start_time' => $start_time]
    );

    while ($record = $cost_query->fetchAssoc()) {
      $metadata = json_decode($record['metadata'], TRUE);
      $total_tokens += $metadata['tokens_used'] ?? 0;
      $total_cost += $metadata['cost_estimate'] ?? 0;
      $total_time += $metadata['processing_time'] ?? 0;
    }

    return [
      'period_days' => $days,
      'total_requests' => (int) $total_requests,
      'completed_requests' => (int) ($status_stats['completed'] ?? 0),
      'failed_requests' => (int) ($status_stats['error'] ?? 0),
      'success_rate' => $total_requests > 0 ? round(($status_stats['completed'] ?? 0) / $total_requests * 100, 2) : 0,
      'task_breakdown' => $task_stats,
      'status_breakdown' => $status_stats,
      'daily_usage' => $daily_usage,
      'totals' => [
        'tokens' => (int) $total_tokens,
        'cost' => round($total_cost, 4),
        'processing_time' => round($total_time, 2),
      ],
      'averages' => [
        'tokens_per_request' => $status_stats['completed'] ?? 0 > 0 ? round($total_tokens / $status_stats['completed'], 2) : 0,
        'cost_per_request' => $status_stats['completed'] ?? 0 > 0 ? round($total_cost / $status_stats['completed'], 4) : 0,
        'time_per_request' => $status_stats['completed'] ?? 0 > 0 ? round($total_time / $status_stats['completed'], 2) : 0,
      ],
    ];
  }

  /**
   * Get global usage statistics.
   *
   * @param int $days
   *   Number of days to look back.
   *
   * @return array
   *   Global usage statistics.
   */
  public function getGlobalStats($days = 30)
  {
    $start_time = $this->time->getRequestTime() - ($days * 24 * 60 * 60);

    // Total requests across all users.
    $total_requests = $this->database->query(
      "SELECT COUNT(*) FROM {sphoenix_ai_usage} WHERE created >= :start_time",
      [':start_time' => $start_time]
    )->fetchField();

    // Active users.
    $active_users = $this->database->query(
      "SELECT COUNT(DISTINCT uid) FROM {sphoenix_ai_usage} WHERE created >= :start_time",
      [':start_time' => $start_time]
    )->fetchField();

    // Most popular tasks.
    $popular_tasks = $this->database->query(
      "SELECT task_type, COUNT(*) as count
       FROM {sphoenix_ai_usage}
       WHERE created >= :start_time AND status = 'completed'
       GROUP BY task_type
       ORDER BY count DESC
       LIMIT 10",
      [':start_time' => $start_time]
    )->fetchAllKeyed();

    // Top users.
    $top_users = $this->database->query(
      "SELECT u.uid, u.name, COUNT(acu.id) as usage_count
       FROM {sphoenix_ai_usage} acu
       JOIN {users_field_data} u ON u.uid = acu.uid
       WHERE acu.created >= :start_time
       GROUP BY u.uid, u.name
       ORDER BY usage_count DESC
       LIMIT 10",
      [':start_time' => $start_time]
    )->fetchAll();

    // Error rate.
    $error_count = $this->database->query(
      "SELECT COUNT(*) FROM {sphoenix_ai_usage} WHERE created >= :start_time AND status = 'error'",
      [':start_time' => $start_time]
    )->fetchField();

    $error_rate = $total_requests > 0 ? round($error_count / $total_requests * 100, 2) : 0;

    // Hourly usage pattern (last 24 hours).
    $hourly_usage = [];
    for ($i = 23; $i >= 0; $i--) {
      $hour_start = $this->time->getRequestTime() - ($i * 60 * 60);
      $hour_start = mktime(date('H', $hour_start), 0, 0, date('m', $hour_start), date('d', $hour_start), date('Y', $hour_start));
      $hour_end = $hour_start + (60 * 60);

      $count = $this->database->query(
        "SELECT COUNT(*) FROM {sphoenix_ai_usage} WHERE created >= :start AND created < :end",
        [':start' => $hour_start, ':end' => $hour_end]
      )->fetchField();

      $hourly_usage[date('H:i', $hour_start)] = (int) $count;
    }

    return [
      'period_days' => $days,
      'total_requests' => (int) $total_requests,
      'active_users' => (int) $active_users,
      'error_rate' => $error_rate,
      'popular_tasks' => $popular_tasks,
      'top_users' => $top_users,
      'hourly_usage' => $hourly_usage,
    ];
  }

  /**
   * Get recent usage activity.
   *
   * @param int $limit
   *   Maximum number of records to return.
   * @param int $uid
   *   Optional user ID to filter by.
   *
   * @return array
   *   Recent activity records.
   */
  public function getRecentActivity($limit = 50, $uid = NULL)
  {
    $query = $this->database->select('sphoenix_ai_usage', 'acu')
      ->fields('acu')
      ->orderBy('created', 'DESC')
      ->range(0, $limit);

    $query->leftJoin('users_field_data', 'u', 'u.uid = acu.uid');
    $query->addField('u', 'name', 'username');

    if ($uid) {
      $query->condition('acu.uid', $uid);
    }

    $results = $query->execute()->fetchAll();

    // Process metadata.
    foreach ($results as &$record) {
      $record->metadata = json_decode($record->metadata, TRUE) ?: [];
      $record->created_formatted = date('Y-m-d H:i:s', $record->created);
    }

    return $results;
  }

  /**
   * Calculate the length of AI response.
   *
   * @param array $response_data
   *   The AI response data.
   *
   * @return int
   *   The calculated response length.
   */
  protected function calculateResponseLength(array $response_data)
  {
    $length = 0;

    if (isset($response_data['content'])) {
      $content = $response_data['content'];
      $length += strlen($content['title'] ?? '');
      $length += strlen($content['html_content'] ?? '');
      $length += strlen($content['plain_text'] ?? '');
      $length += strlen($content['meta_description'] ?? '');
    }

    if (isset($response_data['analysis']['analysis_text'])) {
      $length += strlen($response_data['analysis']['analysis_text']);
    }

    return $length;
  }

  /**
   * Clean up old usage data.
   *
   * @param int $days
   *   Remove records older than this many days.
   *
   * @return int
   *   Number of records deleted.
   */
  public function cleanupOldData($days = 365)
  {
    $cutoff_time = $this->time->getRequestTime() - ($days * 24 * 60 * 60);

    return $this->database->delete('sphoenix_ai_usage')
      ->condition('created', $cutoff_time, '<')
      ->execute();
  }

  /**
   * Export usage data for a user.
   *
   * @param int $uid
   *   The user ID.
   * @param int $days
   *   Number of days to export.
   *
   * @return array
   *   Exportable usage data.
   */
  public function exportUserData($uid, $days = 30)
  {
    $start_time = $this->time->getRequestTime() - ($days * 24 * 60 * 60);

    $query = $this->database->select('sphoenix_ai_usage', 'acu')
      ->fields('acu')
      ->condition('uid', $uid)
      ->condition('created', $start_time, '>=')
      ->orderBy('created', 'DESC');

    $results = $query->execute()->fetchAll();

    // Process and sanitize data.
    $export_data = [];
    foreach ($results as $record) {
      $metadata = json_decode($record->metadata, TRUE) ?: [];

      // Remove sensitive information.
      unset($metadata['error_message']);
      unset($metadata['node_id']);

      $export_data[] = [
        'date' => date('Y-m-d H:i:s', $record->created),
        'task_type' => $record->task_type,
        'status' => $record->status,
        'tokens_used' => $metadata['tokens_used'] ?? 0,
        'processing_time' => $metadata['processing_time'] ?? 0,
        'humanization_applied' => $metadata['humanization_applied'] ?? FALSE,
      ];
    }

    return $export_data;
  }
}
