<?php

namespace Drupal\spotify_embed\Plugin\Block;

use Drupal\Core\Block\BlockBase;
use Drupal\Core\Form\FormStateInterface;
use GuzzleHttp\Client;

/**
 * Provides a Spotify Embed block.
 *
 * @Block(
 *   id = "spotify_embed_block",
 *   admin_label = @Translation("Spotify Embed Block")
 * )
 */
class SpotifyEmbedBlock extends BlockBase {

  public function defaultConfiguration() {
    return [
      'client_id' => '',
      'client_secret' => '',
      'spotify_id' => '',
      'content_type' => 'playlist',
    ];
  }

  public function blockForm($form, FormStateInterface $form_state) {
    $form['client_id'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Spotify Client ID'),
      '#default_value' => $this->configuration['client_id'],
      '#required' => TRUE,
    ];
    $form['client_secret'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Spotify Secret ID'),
      '#default_value' => $this->configuration['client_secret'],
      '#required' => TRUE,
    ];
    $form['spotify_id'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Show ID / Playlist ID'),
      '#default_value' => $this->configuration['spotify_id'],
      '#required' => TRUE,
    ];
    $form['content_type'] = [
      '#type' => 'radios',
      '#title' => $this->t('Content Type'),
      '#options' => [
        'playlist' => $this->t('Playlist'),
        'podcast' => $this->t('Podcast'),
      ],
      '#default_value' => $this->configuration['content_type'],
    ];

    return $form;
  }

  public function blockSubmit($form, FormStateInterface $form_state) {
    $this->configuration['client_id'] = $form_state->getValue('client_id');
    $this->configuration['client_secret'] = $form_state->getValue('client_secret');
    $this->configuration['spotify_id'] = $form_state->getValue('spotify_id');
    $this->configuration['content_type'] = $form_state->getValue('content_type');
  }

  public function build() {
    $id = $this->configuration['spotify_id'];
    $type = $this->configuration['content_type'];
    $client_id = $this->configuration['client_id'];
    $client_secret = $this->configuration['client_secret'];

    $items = $this->fetchSpotifyItems($client_id, $client_secret, $id, $type);

    if (empty($items)) {
      return [
        '#markup' => '<p>No items returned from Spotify API. Check your credentials or ID.</p>',
      ];
    }

    return [
      '#theme' => 'spotify_embed',
      '#items' => $items,
      '#type' => $type,
      '#attached' => [
        'library' => ['spotify_embed/slider'],
      ],
    ];
  }

  /**
   * Fetch tracks or episodes from Spotify API.
   */
  private function fetchSpotifyItems($client_id, $client_secret, $id, $type) {
    $client = new Client();

    // Step 1: Get token.
    $auth = base64_encode("$client_id:$client_secret");
    $response = $client->post('https://accounts.spotify.com/api/token', [
      'headers' => [
        'Authorization' => "Basic $auth",
      ],
      'form_params' => [
        'grant_type' => 'client_credentials',
      ],
    ]);
    $data = json_decode($response->getBody(), TRUE);
    if (empty($data['access_token'])) {
      return [];
    }
    $token = $data['access_token'];

    $items = [];

    // Step 2: Fetch playlist tracks or show episodes.
    if ($type === 'playlist') {
      $url = "https://api.spotify.com/v1/playlists/$id/tracks?limit=50";
      $response = $client->get($url, [
        'headers' => ['Authorization' => "Bearer $token"],
      ]);
      $data = json_decode($response->getBody(), TRUE);
      foreach ($data['items'] as $track) {
        if (!empty($track['track']['id'])) {
          $items[] = $track['track']['id'];
        }
      }
    }
    else {
      $url = "https://api.spotify.com/v1/shows/$id/episodes?limit=50";
      $response = $client->get($url, [
        'headers' => ['Authorization' => "Bearer $token"],
      ]);
      $data = json_decode($response->getBody(), TRUE);
      foreach ($data['items'] as $episode) {
        if (!empty($episode['id'])) {
          $items[] = $episode['id'];
        }
      }
    }

    return $items;
  }
}
