<?php

declare(strict_types=1);

namespace Drupal\sql_views_examples\Plugin\SqlView;

use Drupal\Core\Database\Query\SelectInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup as TM;
use Drupal\sql_views\Attribute\SqlView;
use Drupal\sql_views\SqlViewPluginBase;

/**
 * Plugin implementation of the sql_view.
 */
#[SqlView(
  id: 'top_authors',
  label: new TM('Top Authors'),
)]
final class TopAuthors extends SqlViewPluginBase {

  /**
   * {@inheritdoc}
   */
  public function getQuery(): SelectInterface {
    $query = $this->database->select('users', 'u');
    $query->fields('u', ['uid']);
    $query->condition('u.uid', 0, '>');
    $query->groupBy('u.uid');
    $query->addExpression('COUNT(*)', 'total');
    $query->leftJoin('node_field_data', 'nfd', 'nfd.uid = u.uid');
    $query->orderBy('total', 'DESC');
    return $query;
  }

  /**
   * {@inheritdoc}
   */
  public function getViewsData(): array {
    $data['table']['group'] = $this->pluginDefinition['label'];

    $data['table']['base'] = [
      'field' => 'uid',
      'title' => $this->pluginDefinition['label'],
      'help' => new TM('Top authors'),
    ];

    $data['table']['join']['users_field_data'] = [
      'label' => 'User',
      'left_field' => 'uid',
      'field' => 'uid',
    ];

    $data['uid'] = [
      'title' => new TM('UID'),
      'help' => new TM('User ID'),
      'field' => ['id' => 'standard'],
      'sort' => ['id' => 'standard'],
      'filter' => ['id' => 'string'],
      'argument' => ['id' => 'string'],
      'relationship' => [
        'title' => new TM('User'),
        'base' => 'users_field_data',
        'base field' => 'uid',
        'id' => 'standard',
      ],
    ];

    $data['total'] = [
      'title' => new TM('Total'),
      'help' => new TM('Total count of nodes created by the author'),
      'field' => ['id' => 'numeric'],
      'sort' => ['id' => 'standard'],
      'filter' => ['id' => 'numeric'],
      'argument' => ['id' => 'numeric'],
    ];

    return $data;
  }

}
