<?php

declare(strict_types=1);

namespace Drupal\sql_views\Command;

use Drupal\Core\Database\Connection;
use Drupal\sql_views\SqlViewPluginManager;
use Symfony\Component\Console\Attribute\AsCommand;
use Symfony\Component\Console\Command\Command;
use Symfony\Component\Console\Helper\TableCell;
use Symfony\Component\Console\Helper\TableCellStyle;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Output\OutputInterface;
use Symfony\Component\Console\Style\SymfonyStyle;

/**
 * {@selfdoc}
 */
#[AsCommand(
  name: 'sql:views:list',
  description: 'Lists SQL views',
)]
final class ListCommand extends Command {

  /**
   * {@inheritdoc}
   */
  public function __construct(
    private readonly Connection $database,
    private readonly SqlViewPluginManager $viewManager,
  ) {
    parent::__construct();
  }

  /**
   * @return int<0, 0>
   */
  protected function execute(InputInterface $input, OutputInterface $output): int {
    $io = new SymfonyStyle($input, $output);

    $table = $io->createTable();
    $table->setHeaders(['ID', 'Label', 'Installed', 'Records']);
    $total_options = ['style' => new TableCellStyle(['align' => 'right'])];
    $status_options = ['style' => new TableCellStyle(['align' => 'center'])];

    foreach ($this->viewManager->getSupportedDefinitions() as $plugin_id => $definition) {
      $view_name = $this->viewManager->getViewName($plugin_id);
      $is_installed = $this->database->schema()->tableExists($view_name);
      $total = '';
      if ($is_installed) {
        $total = $this->database
          ->select($view_name)
          ->countQuery()
          ->execute()
          ->fetchField();
      }
      $row = [
        $plugin_id,
        $definition['label'],
        new TableCell($is_installed ? 'Yes' : 'No', $status_options),
        new TableCell($total, $total_options),
      ];
      $table->addRow($row);
    }

    $table->render();

    return self::SUCCESS;
  }

}
