<?php

declare(strict_types=1);

namespace Drupal\sql_views;

use Drupal\Core\Cache\CacheBackendInterface;
use Drupal\Core\Cache\CacheTagsInvalidator;
use Drupal\Core\Database\Connection;
use Drupal\Core\Database\Query\SelectInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Plugin\DefaultPluginManager;
use Drupal\sql_views\Attribute\SqlView;

/**
 * A plugin manager for SQL views.
 */
final class SqlViewPluginManager extends DefaultPluginManager {

  public const SQL_VIEW_PREFIX = 'sv_';

  /**
   * {@selfdoc}
   */
  public function __construct(\Traversable $namespaces, CacheBackendInterface $cache_backend, ModuleHandlerInterface $module_handler, private Connection $database, private CacheTagsInvalidator $cacheTagsInvalidator) {
    parent::__construct('Plugin/SqlView', $namespaces, $module_handler, SqlViewInterface::class, SqlView::class);
    $this->alterInfo('sql_view_info');
    $this->setCacheBackend($cache_backend, 'sql_view_plugins', ['sql_view_plugins']);
  }

  /**
   * @return array<string, array{id: string, label: \Drupal\Core\StringTranslation\TranslatableMarkup, provider: string, supportedDrivers: list<string>}>
   */
  public function getSupportedDefinitions(): array {
    $driver = $this->database->driver();
    $definitions = [];
    foreach ($this->getDefinitions() as $plugin_id => $definition) {
      if ($definition['supportedDrivers'] === NULL || \in_array($driver, $definition['supportedDrivers'], TRUE)) {
        $definitions[$plugin_id] = $definition;
      }
    }
    return $definitions;
  }

  /**
   * {@selfdoc}
   */
  public function rebuild(): void {
    $this->cacheTagsInvalidator->invalidateTags($this->getCacheTags());
    foreach ($this->getSupportedDefinitions() as $plugin_id => $definition) {
      // PosgreSQL does not permit replacing views with different types of
      // fields. Also, SQLite does not support the CREATE OR REPLACE syntax. So
      // we drop the view manually before installing it.
      $this->uninstallView($plugin_id);
      $this->installView($plugin_id);
    }
  }

  /**
   * {@selfdoc}
   */
  public function installView(string $plugin_id): void {
    $query = $this->createInstance($plugin_id)->getQuery();
    $this->database->query(
      \sprintf('CREATE VIEW %s AS %s', self::getViewName($plugin_id), $query),
      $query instanceof SelectInterface ? $query->getArguments() : [],
    );
  }

  /**
   * {@selfdoc}
   */
  public function uninstallView(string $plugin_id): void {
    $this->database->query(
      \sprintf('DROP VIEW IF EXISTS %s', self::getViewName($plugin_id)),
    );
  }

  /**
   * {@selfdoc}
   */
  public static function getViewName(string $plugin_id): string {
    return self::SQL_VIEW_PREFIX . $plugin_id;
  }

}
