<?php

namespace Drupal\sqlite_backup\Form;

use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Symfony\Component\HttpKernel\Exception\AccessDeniedHttpException;

/**
 * Clear caches for this site.
 */
class CacheClearForm extends FormBase {

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'sqlite_backup_clear_cache';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    if ($this->currentUser()->isAnonymous()) {
      // Most likely the user session expired since the backup.
      // redirect to the login screen with a redirect destination to come back.
      // @phpstan-ignore return.type
      return $this->redirect('user.login', [], ['query' => $this->getRedirectDestination()->getAsArray()]);
    }
    if (!$this->currentUser()->hasPermission('administer sqlite backups')) {
      // No access for authenticated users without the permission.
      // This could happen if the user gained the permissions in the database
      // after the backup was created before restoring it and the session is
      // still the same. Most likely, however, it is just a random user without
      // the privilege. We allow all users on the route so that we can do the
      // redirect above and provide a good user experience.
      throw new AccessDeniedHttpException();
    }

    $form['#title'] = $this->t('Clear cache.');

    $form['#attributes']['class'][] = 'confirmation';
    $form['description'] = [
      '#markup' => $this->t('The database has been restored, clear all caches to also restore the cache.'),
    ];
    $form['confirm'] = ['#type' => 'hidden', '#value' => 1];

    $form['actions'] = ['#type' => 'actions'];
    $form['actions']['submit'] = [
      '#type' => 'submit',
      '#value' => $this->t('Clear cache'),
      '#button_type' => 'primary',
    ];

    $form['#theme'] = 'confirm_form';

    return $form;
  }

  /**
   * Clears the caches.
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    drupal_flush_all_caches();
    $this->messenger()->addStatus($this->t('Caches cleared.'));
    $form_state->setRedirect('sqlite_backup.overview');
  }

}
