<?php

declare(strict_types=1);

namespace Drupal\sqlite_backup\Form;

use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\sqlite_backup\SqliteBackupManager;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides a SQLite Backup form.
 */
final class CreateBackupForm extends FormBase {

  /**
   * The backup manager.
   *
   * @var \Drupal\sqlite_backup\SqliteBackupManager
   */
  private SqliteBackupManager $manager;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    $form = parent::create($container);
    $form->manager = $container->get(SqliteBackupManager::class);
    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'sqlite_backup_create';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {

    if (!$this->manager->isSqlite()) {
      $build['sorry'] = [
        '#type' => 'markup',
        '#markup' => $this->t('Backups can only be created when the database is sqlite.'),
      ];

      return $build;
    }

    $form['label'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Label'),
      '#description' => $this->t('The label of the backup to create.'),
      '#maxlength' => 255,
      '#required' => TRUE,
    ];

    $form['actions'] = [
      '#type' => 'actions',
      'submit' => [
        '#type' => 'submit',
        '#value' => $this->t('Create'),
      ],
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state): void {
    // @todo Validate the form here.
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    $new = $this->manager->createBackup($form_state->getValue('label'));

    $this->messenger()->addStatus($this->t('Created %label backup.', ['%label' => $new->label]));
    $form_state->setRedirect('sqlite_backup.overview');
  }

}
