<?php

namespace Drupal\Tests\sqlite_backup\Kernel;

use Drupal\Core\Site\Settings;
use Drupal\KernelTests\KernelTestBase;
use Drupal\sqlite\Driver\Database\sqlite\Connection;
use Drupal\sqlite_backup\SqliteBackupManager;
use Drupal\Tests\Core\Database\Stub\StubPDO;

/**
 * Kernel test for the backup manager.
 */
class BackupManagerTest extends KernelTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = ['sqlite_backup'];

  /**
   * {@inheritdoc}
   */
  protected $defaultTheme = 'stark';

  /**
   * Test how creating and reverting works.
   */
  public function testRevert(): void {

    $databaseOptions = [
      'database' => Settings::get('file_public_path') . '/.ht.test.sqlite',
    ];
    $pdo = $this->createMock(StubPDO::class);

    $connection = new Connection($pdo, $databaseOptions);
    $manager = new SqliteBackupManager(
      $connection,
      $this->container->get('file_system'),
      $this->container->get('datetime.time'),
      $this->container->get('transliteration'),
      $this->container->get('uuid'),
    );

    self::assertTrue($manager->isSqlite());

    // Create a "database" the backup manager doesn't care that it is not
    // actually a database, so we just put random content we can verify.
    file_put_contents($databaseOptions['database'], 'initial data');

    $firstBackup = $manager->createBackup('first');

    // Write new data to the database.
    file_put_contents($databaseOptions['database'], 'new data');

    $secondBackup = $manager->createBackup('second');

    // Write new data to the database.
    file_put_contents($databaseOptions['database'], 'more new data');

    // Restore the backup, this does not affect the database yet.
    $manager->restoreBackup($firstBackup);
    self::assertEquals('more new data', file_get_contents($databaseOptions['database']));

    // We can destruct the manager but the revert operation still waits.
    unset($manager);
    self::assertEquals('more new data', file_get_contents($databaseOptions['database']));

    // But if the database is no longer used it is now safe to revert.
    unset($connection);
    self::assertEquals('initial data', file_get_contents($databaseOptions['database']));

    // We can create the manager again and it will read the available data.
    $connection = new Connection($pdo, $databaseOptions);
    $manager = new SqliteBackupManager(
      $connection,
      $this->container->get('file_system'),
      $this->container->get('datetime.time'),
      $this->container->get('transliteration'),
      $this->container->get('uuid'),
    );

    self::assertCount(2, $manager->getBackups());

    // Test removing a backup.
    $manager->deleteBackup($firstBackup->id);
    $backups = $manager->getBackups();
    self::assertEquals([$secondBackup->id => $secondBackup], $backups);

    // Restore the second backup.
    $manager->restoreBackup($secondBackup);
    self::assertEquals('initial data', file_get_contents($databaseOptions['database']));

    unset($manager);
    unset($connection);
    self::assertEquals('new data', file_get_contents($databaseOptions['database']));
  }

}
