<?php

declare(strict_types=1);

namespace Drupal\sqlite_backup\Controller;

use Drupal\Component\Serialization\Json;
use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Datetime\DateFormatterInterface;
use Drupal\Core\Url;
use Drupal\sqlite_backup\SqliteBackup;
use Drupal\sqlite_backup\SqliteBackupManager;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Returns responses for SQLite Backup routes.
 */
final class SqliteBackupController extends ControllerBase {

  /**
   * The controller constructor.
   */
  public function __construct(
    private readonly SqliteBackupManager $manager,
    private DateFormatterInterface $dateFormatter,
  ) {}

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): self {
    return new self(
      $container->get(SqliteBackupManager::class),
      $container->get(DateFormatterInterface::class),
    );
  }

  /**
   * Builds the response.
   */
  public function __invoke(): array {

    if (!$this->manager->isSqlite()) {
      return $this->sorryNotSqlite();
    }

    $header = [
      'label' => $this->t('Name'),
      'date' => $this->t('Date'),
      'operations' => $this->t('Operations'),
    ];

    $build['warning'] = [
      '#type' => 'markup',
      '#markup' => $this->t('Be careful when changing the codebase and restoring older backups. Restoring old backups may break the site and may need database updates to be run in order to work again.'),
    ];

    $build['table'] = [
      '#type' => 'table',
      '#header' => $header,
      '#title' => $this->t('Database backups'),
      '#rows' => [],
      '#empty' => $this->t('There are no database backups yet.'),
      '#cache' => [
        // @todo find out how to cache this list.
        // 'contexts' => [],
        // 'tags' => [],
        'max-age' => 0,
      ],
    ];

    $backups = [];
    foreach ($this->manager->getBackups() as $backup) {
      $backups[$backup->timestamp] = $backup;
    }
    krsort($backups);

    foreach ($backups as $backup) {
      $build['table']['#rows'][$backup->id] = [
        'label' => $backup->label,
        'date' => $this->dateFormatter->format($backup->timestamp),
        'operations' => [
          'data' => [
            '#type' => 'operations',
            '#links' => $this->getOperations($backup),
            // Allow links to use modals.
            '#attached' => [
              'library' => ['core/drupal.dialog.ajax'],
            ],
          ],
        ],
      ];
    }

    return $build;
  }

  /**
   * Get the operations available on a backup.
   *
   * @param \Drupal\sqlite_backup\SqliteBackup $backup
   *   The backup.
   *
   * @return array
   *   The render array representing the options on a backup.
   */
  public function getOperations(SqliteBackup $backup) {

    $operations = [];

    // Do not set the redirect url.
    $revert_url = Url::fromRoute('sqlite_backup.restore', ['id' => $backup->id]);
    $attributes = $revert_url->getOption('attributes') ?: [];
    $attributes += ['aria-label' => $this->t('Revert to backup @label', ['@label' => $backup->label])];
    $revert_url->setOption('attributes', $attributes);

    $operations['revert'] = [
      'title' => $this->t('Revert'),
      'weight' => 10,
      'url' => $revert_url,
    ];

    $delete_url = $this->ensureDestination(Url::fromRoute('sqlite_backup.delete', ['id' => $backup->id]));
    $attributes = $delete_url->getOption('attributes') ?: [];
    $attributes += ['aria-label' => $this->t('Delete backup @label', ['@label' => $backup->label])];
    $delete_url->setOption('attributes', $attributes);

    $operations['delete'] = [
      'title' => $this->t('Delete'),
      'weight' => 100,
      'attributes' => [
        'class' => ['use-ajax'],
        'data-dialog-type' => 'modal',
        'data-dialog-options' => Json::encode([
          'width' => 880,
        ]),
      ],
      'url' => $delete_url,
    ];

    return $operations;
  }

  /**
   * Get the non-sqlite response.
   *
   * @return array
   *   The render array for cases when the database is not sqlite.
   */
  protected function sorryNotSqlite(): array {
    $build['sorry'] = [
      '#type' => 'markup',
      '#markup' => $this->t('This module can only be used when the database is sqlite.'),
    ];

    return $build;
  }

  /**
   * Ensures that a destination is present on the given URL.
   *
   * @param \Drupal\Core\Url $url
   *   The URL object to which the destination should be added.
   *
   * @return \Drupal\Core\Url
   *   The updated URL object.
   */
  private function ensureDestination(Url $url) {
    return $url->mergeOptions(['query' => $this->getRedirectDestination()->getAsArray()]);
  }

}
