<?php

declare(strict_types=1);

namespace Drupal\sqlite_backup\Form;

use Drupal\Core\Datetime\DateFormatterInterface;
use Drupal\Core\Form\ConfirmFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\Core\Url;
use Drupal\sqlite_backup\SqliteBackup;
use Drupal\sqlite_backup\SqliteBackupManager;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpKernel\Exception\NotFoundHttpException;

/**
 * Provides a SQLite Delete form.
 */
final class DeleteForm extends ConfirmFormBase {

  /**
   * The backup manager.
   *
   * @var \Drupal\sqlite_backup\SqliteBackupManager
   */
  private SqliteBackupManager $manager;

  /**
   * The date formatter.
   *
   * @var \Drupal\Core\Datetime\DateFormatterInterface
   */
  private DateFormatterInterface $dateFormatter;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    $form = parent::create($container);
    $form->manager = $container->get(SqliteBackupManager::class);
    $form->dateFormatter = $container->get(DateFormatterInterface::class);
    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'sqlite_backup_delete';
  }

  /**
   * {@inheritdoc}
   */
  public function getQuestion(): TranslatableMarkup {
    return $this->t('Are you sure you want to delete the database backup?');
  }

  /**
   * {@inheritdoc}
   */
  public function getDescription() {
    // This throws an exception if the backup does not exist.
    $backup = $this->getBackup();

    return $this->t('Delete the "%name" backup from %date. This can not be undone.', [
      '%name' => $backup->label,
      '%date' => $this->dateFormatter->format($backup->timestamp),
    ]);
  }

  /**
   * {@inheritdoc}
   */
  public function getCancelUrl(): Url {
    return new Url('sqlite_backup.overview');
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    $backup = $this->getBackup();

    $this->manager->deleteBackup($backup->id);

    $this->messenger()->addStatus($this->t('The backup %name has been deleted.', ['%name' => $backup->label]));
    $form_state->setRedirect('sqlite_backup.overview');
  }

  /**
   * Get the backup from the route.
   *
   * @return \Drupal\sqlite_backup\SqliteBackup
   *   The backup value object.
   */
  protected function getBackup(): SqliteBackup {
    $id = $this->getRouteMatch()->getRawParameter('id');

    $all = $this->manager->getBackups();
    if (!isset($all[$id])) {
      throw new NotFoundHttpException();
    }

    return $all[$id];
  }

}
