<?php

declare(strict_types=1);

namespace Drupal\sqlite_backup\Form;

use Drupal\Core\Ajax\AjaxResponse;
use Drupal\Core\Ajax\MessageCommand;
use Drupal\Core\Ajax\RedirectCommand;
use Drupal\Core\Ajax\RemoveCommand;
use Drupal\Core\Datetime\DateFormatterInterface;
use Drupal\Core\Form\ConfirmFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\Core\Url;
use Drupal\sqlite_backup\Ajax\WaitCommand;
use Drupal\sqlite_backup\SqliteBackup;
use Drupal\sqlite_backup\SqliteBackupManager;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpKernel\Exception\NotFoundHttpException;

/**
 * Provides a SQLite Backup form.
 */
final class RestoreForm extends ConfirmFormBase {

  /**
   * The backup manager.
   *
   * @var \Drupal\sqlite_backup\SqliteBackupManager
   */
  private SqliteBackupManager $manager;

  /**
   * The date formatter.
   *
   * @var \Drupal\Core\Datetime\DateFormatterInterface
   */
  private DateFormatterInterface $dateFormatter;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    $form = parent::create($container);
    $form->manager = $container->get(SqliteBackupManager::class);
    $form->dateFormatter = $container->get(DateFormatterInterface::class);
    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'sqlite_backup_restore';
  }

  /**
   * {@inheritdoc}
   */
  public function getQuestion(): TranslatableMarkup {
    return $this->t('Are you sure you want to restore the database backup?');
  }

  /**
   * {@inheritdoc}
   */
  public function getDescription() {
    // This throws an exception if the backup does not exist.
    $backup = $this->getBackup();

    return $this->t('Revert to the "%name" backup from %date. <br/>
    This can not be undone. After reverting to the backup you need to clear all caches. <br/>
    This action may log you out. If you are logged out you still need to clear caches after you log in again.<br/>
    Backups should only be reverted when there are no other visitors to the site, you may consider enabling the <a href="@maintenance-mode">maintenance mode</a>', [
      '%name' => $backup->label,
      '%date' => $this->dateFormatter->format($backup->timestamp),
      '@maintenance-mode' => Url::fromRoute('system.site_maintenance_mode')->toString(),
    ]);
  }

  /**
   * {@inheritdoc}
   */
  public function getCancelUrl(): Url {
    return new Url('sqlite_backup.overview');
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $form = parent::buildForm($form, $form_state);

    $form['actions']['submit']['#ajax'] = [
      'callback' => '::ajaxSubmitForm',
      'event' => 'click',
      'progress' => [
        'type' => 'throbber',
        'message' => $this->t('Please wait'),
      ],
    ];
    $form['#attached']['library'][] = "sqlite_backup/sqlite_backup";

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function ajaxSubmitForm(array &$form, FormStateInterface $form_state) {
    $backup = $this->getBackup();
    // Restore the backup, this will take effect in the shutdown phase of php.
    $this->manager->restoreBackup($backup);

    // Issue the redirect command after a second of timeout to allow the files
    // to be copied.
    $response = new AjaxResponse();
    $response->addCommand(new RemoveCommand('#edit-actions'));
    $response->addCommand(new MessageCommand('Redirecting to the cache clearing form.'));
    $response->addCommand(new WaitCommand(1));
    $response->addCommand(new RedirectCommand(Url::fromRoute('sqlite_backup.cache_clear')->toString()));
    return $response;
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    // The form was submitted via ajax.
  }

  /**
   * Get the backup from the route.
   *
   * @return \Drupal\sqlite_backup\SqliteBackup
   *   The backup value object.
   */
  protected function getBackup(): SqliteBackup {
    $id = $this->getRouteMatch()->getRawParameter('id');

    $all = $this->manager->getBackups();
    if (!isset($all[$id])) {
      throw new NotFoundHttpException();
    }

    return $all[$id];
  }

}
