<?php

namespace Drupal\sqlite_backup\Hook;

use Drupal\Core\File\FileSystemInterface;
use Drupal\Core\Routing\RouteMatchInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\sqlite_backup\SqliteBackupManager;
use Drupal\Core\Hook\Attribute\Hook;

/**
 * Help hooks for our module.
 */
class HelpHooks {

  use StringTranslationTrait;

  public function __construct(
    private readonly SqliteBackupManager $backupManager,
    private readonly FileSystemInterface $fileSystem,
  ) {}

  /**
   * Implements hook_help().
   */
  #[Hook('help')]
  public function help($route_name, RouteMatchInterface $route_match) : string|\Stringable|array|null {
    if (!$this->backupManager->isSqlite()) {
      return str_starts_with($route_name, 'sqlite_backup') ? '<p>' . $this->t('When setting up Drupal, different database technologies can be selected. This module only works when SQLite was selected.') . '</p>' : NULL;
    }

    switch ($route_name) {
      case 'sqlite_backup.overview':
        return '<p>' . $this->t('The backups are in the same folder as the active SQLite database: @folder', [
          '@folder' => $this->fileSystem->realpath($this->backupManager->getDatabaseFolder()),
        ]) . '<br/>' . $this->t('The file name contains the backup name and a random UUID. The backup metadata is saved in @filename in the same folder.', [
          '@filename' => SqliteBackupManager::BACKUPS_FILE,
        ]) . '</p>';

      case 'sqlite_backup.create':
        return '<p>' . $this->t('Backups can not be renamed from the UI.') . '</p>';

      case 'sqlite_backup.restore':
        $id = $route_match->getRawParameter('id');
        $backups = $this->backupManager->getBackups();
        if (!isset($backups[$id])) {
          return NULL;
        }

        return '<p>' . $this->t('In rare cases it can happen that restoration of a backup can cause a corrupted database. <br/> If this happens the easiest is to manually replace the database file (@active_file) with the one of the backup (@backup_file)', [
          '@active_file' => $this->backupManager->getDatabasePath(),
          '@backup_file' => $backups[$id]->file,
        ]) . '</p>';
    }
    return NULL;
  }

}
