<?php

declare(strict_types=1);

namespace Drupal\sqlite_backup;

use Drupal\Component\Datetime\TimeInterface;
use Drupal\Component\Transliteration\TransliterationInterface;
use Drupal\Component\Uuid\UuidInterface;
use Drupal\Core\Database\Connection;
use Drupal\Core\File\FileExists;
use Drupal\sqlite\Driver\Database\sqlite\Connection as SqliteConnection;
use Drupal\Core\File\FileSystemInterface;

/**
 * Internal API to manage the sqlite backups.
 *
 * @internal This is not a public API (yet) it may change at any moment.
 */
final class SqliteBackupManager {

  /**
   * The filename of the management file.
   */
  public const string BACKUPS_FILE = '.ht.sqlite_backups.json';

  /**
   * Constructs a SqliteBackupManager object.
   *
   * @param \Drupal\Core\Database\Connection $connection
   *   The database connection for the site uses sqlite and the file location.
   * @param \Drupal\Core\File\FileSystemInterface $fileSystem
   *   The filesystem to manage the backups and the actual database file.
   * @param \Drupal\Component\Datetime\TimeInterface $time
   *   The time service.
   * @param \Drupal\Component\Transliteration\TransliterationInterface $transliteration
   *   The transliteration service to create a machine name out of the name.
   * @param \Drupal\Component\Uuid\UuidInterface $uuid
   *   The uuid generator.
   */
  public function __construct(
    private readonly Connection $connection,
    private readonly FileSystemInterface $fileSystem,
    private readonly TimeInterface $time,
    private readonly TransliterationInterface $transliteration,
    private readonly UuidInterface $uuid,
  ) {}

  /**
   * Check if the module can be used.
   *
   * @return bool
   *   True if the database is on sqlite.
   */
  public function isSqlite(): bool {
    return $this->connection instanceof SqliteConnection;
  }

  /**
   * Create a new backup.
   *
   * @param \Stringable|string $name
   *   The name to give the backup.
   *
   * @return \Drupal\sqlite_backup\SqliteBackup
   *   The backup value object.
   */
  public function createBackup(\Stringable|string $name): SqliteBackup {
    if (!$this->isSqlite()) {
      throw new \BadMethodCallException('Can only create backups when the database is sqlite');
    }
    $id = $this->uuid->generate();
    // Add the id to the filename to avoid overwriting another backup with the
    // same name.
    $fileName = '.ht.' . $this->transliterate((string) $name) . '-' . $id . '.sqlite';
    $path = $this->fileSystem->copy($this->getDatabasePath(), $this->getDatabaseFolder() . $fileName, FileExists::Rename);

    $backup = new SqliteBackup(
      id: $id,
      label: $name,
      timestamp: $this->time->getRequestTime(),
      file: $this->fileSystem->basename($path),
    );

    $backups = $this->getBackups();
    $backups[$id] = $backup;
    $this->saveBackups($backups);

    return $backup;
  }

  /**
   * Restore a backup.
   *
   * @param \Drupal\sqlite_backup\SqliteBackup $backup
   *   The backup to restore.
   */
  public function restoreBackup(SqliteBackup $backup): void {
    if (!$this->isSqlite()) {
      throw new \BadMethodCallException('Can only restore backups when the database is sqlite');
    }

    // Schedule overwriting the active database connection.
    $revert['source'] = $this->getDatabaseFolder() . $backup->file;
    $revert['destination'] = $this->getDatabasePath();
    $revert['real_source'] = $this->fileSystem->realpath($revert['source']);
    $revert['real_destination'] = $this->fileSystem->realpath($revert['destination']);

    // We smuggle an object into the database connection settings.
    // This object will go out of scope when the database is garbage collected.
    // We copy the backup database over the active database when the database is
    // no longer used to avoid pending transactions corrupting the reverted
    // database.
    \Closure::bind(
      function ($revert) {
        // phpcs:disable
        $this->connectionOptions['sqlite_backup_reverter'] = new class($revert) {
          protected static int $count = 0;

          public function __construct(
            protected readonly array $revert,
          ) {
            self::$count++;
          }

          /**
           * The destructor copying the backup over the database.
           */
          public function __destruct() {
            self::$count--;
            if (self::$count > 0) {
              // Make sure that only the last object reverts the database.
              return;
            }
            $source = $this->revert['source'];
            $destination = $this->revert['destination'];
            if (file_exists($source) && file_exists($destination)) {
              // The following is an extract from
              // \Drupal\Core\File\FileSystem::copy()
              if (!@copy($source, $destination)) {
                // If the copy failed and realpaths exist, retry the operation
                // using them instead.
                $real_source = $this->revert['real_source'] ?: $source;
                $real_destination = $this->revert['real_destination'] ?: $destination;
                if ($real_source === FALSE || $real_destination === FALSE || !copy($real_source, $real_destination)) {
                  // Something went wrong, but this is after the database is
                  // already not available anymore. It is unclear what could be
                  // done now. We assume the user finds out by themselves that
                  // the revert was not successful.
                  return;
                }
              }
            }
          }

        };
        // phpcs:enable
      },
      $this->connection,
      Connection::class
    )($revert);

  }

  /**
   * Delete a backup.
   *
   * @param string $id
   *   The id of the backup to delete.
   */
  public function deleteBackup(string $id): void {
    if (!$this->isSqlite()) {
      throw new \BadMethodCallException('Can only delete backups when the database is sqlite');
    }
    $backups = $this->getBackups();
    if (!isset($backups[$id])) {
      throw new \InvalidArgumentException('Backup not found.');
    }
    $backup = $backups[$id];
    $this->fileSystem->delete($this->getDatabaseFolder() . $backup->file);
    unset($backups[$id]);
    $this->saveBackups($backups);
  }

  /**
   * Get the list of backups.
   *
   * @return \Drupal\sqlite_backup\SqliteBackup[]
   *   The existing backups.
   */
  public function getBackups(): array {
    if (!$this->isSqlite()) {
      throw new \BadMethodCallException('Can only read backups when the database is sqlite');
    }
    $file = $this->getDatabaseFolder() . self::BACKUPS_FILE;
    if (!file_exists($file)) {
      return [];
    }
    $json = file_get_contents($file);
    if (empty($json)) {
      return [];
    }
    $data = json_decode($json, TRUE);
    $backups = [];
    foreach ($data as $id => $item) {
      $backup = SqliteBackup::fromArray($id, $item);
      if (file_exists($this->getDatabaseFolder() . $backup->file)) {
        $backups[$id] = $backup;
      }
    }

    return $backups;
  }

  /**
   * Get the filename and path of the sqlite database.
   *
   * @return string
   *   The path.
   */
  public function getDatabasePath(): string {
    return $this->connection->getConnectionOptions()['database'];
  }

  /**
   * Get the directory of where the database is.
   *
   * @return string
   *   The folder to put all the backups.
   */
  public function getDatabaseFolder(): string {
    return $this->fileSystem->dirname($this->getDatabasePath()) . DIRECTORY_SEPARATOR;
  }

  /**
   * Save the backups file.
   *
   * @param \Drupal\sqlite_backup\SqliteBackup[] $backups
   *   The backups to save.
   */
  private function saveBackups(array $backups): void {
    $data = [];
    foreach ($backups as $backup) {
      $data[$backup->id] = $backup->toArray();
    }

    $data = json_encode($data, JSON_HEX_TAG | JSON_HEX_APOS | JSON_HEX_QUOT | JSON_UNESCAPED_UNICODE | JSON_THROW_ON_ERROR | JSON_PRETTY_PRINT);
    $this->fileSystem->saveData($data, $this->getDatabaseFolder() . self::BACKUPS_FILE, FileExists::Replace);
  }

  /**
   * Make a "machine name" out of a given name.
   *
   * @param string $text
   *   The user input with potentially unusable characters.
   *
   * @return string
   *   The machine name.
   */
  private function transliterate(string $text): string {
    $text = $this->transliteration->transliterate($text, 'en', '');
    $text = strtolower($text);
    $text = preg_replace('/[^a-z0-9_]+/', '_', $text);
    $text = preg_replace('/_+/', '_', $text);
    $text = trim($text, '_');
    $text = substr($text, 0, 20);
    return $text;
  }

}
