<?php

namespace Drupal\Tests\sqlite_backup\FunctionalJavascript;

use Drupal\FunctionalJavascriptTests\WebDriverTestBase;
use Drupal\sqlite_backup\SqliteBackupManager;

/**
 * Test the full cycle by reverting to a backup in the UI.
 */
class CreateRevertUITest extends WebDriverTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = ['sqlite_backup'];

  /**
   * {@inheritdoc}
   */
  protected $defaultTheme = 'stark';

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();
    $this->drupalLogin($this->drupalCreateUser([
      'access administration pages',
      'administer site configuration',
      'administer sqlite backups',
      'view the administration theme',
    ]));

    if (!$this->container->get(SqliteBackupManager::class)->isSqlite()) {
      $this->markTestSkipped('Database is not sqlite');
    }

  }

  /**
   * {@inheritdoc}
   */
  protected function tearDown(): void {
    // We remove any leftover backups in case a test fails.
    /** @var \Drupal\sqlite_backup\SqliteBackupManager $manager */
    $manager = $this->container->get(SqliteBackupManager::class);
    if ($manager->isSqlite()) {
      foreach ($manager->getBackups() as $id => $backup) {
        $manager->deleteBackup($id);
      }
    }
    parent::tearDown();
  }

  /**
   * Create a checkpoint and revert to it.
   */
  public function testCreateAndRevertBackup(): void {
    // At the beginning there are no backups.
    $this->drupalGet('/admin/config/development/sqlite-backup');
    $this->assertSession()->pageTextNotContains('Warning message');
    $this->assertSession()->pageTextContains('There are no database backups yet.');

    // So we create a backup.
    $this->drupalGet('/admin/config/development/sqlite-backup/create');
    $this->submitForm(['Label' => 'test post installation backup'], 'Create');
    $this->assertSession()->pageTextContains('Created test post installation backup backup.');
    $backups = $this->container->get(SqliteBackupManager::class)->getBackups();
    self::assertCount(1, $backups);

    // There is only one revert link.
    $this->clickLink('Revert');
    $revertUrl = $this->getUrl();

    // So we change the site name.
    $original = $this->config('system.site')->get('name');
    $this->drupalGet('/admin/config/system/site-information');
    $this->submitForm([
      'Site name' => 'Changed site name',
      // We also change the front page because of no access to /user/login lol.
      'Default front page' => '/admin',
    ], 'Save configuration');
    self::assertEquals('Changed site name', $this->config('system.site')->get('name'));

    // Reload the revert form.
    $this->drupalGet($revertUrl);

    $this->submitForm([], 'Confirm');
    $this->assertSession()->waitForText('The database has been restored, clear all caches to also restore the cache.');
    $connectionOptions = \Drupal::database()->getConnectionOptions();
    $this->submitForm([], 'Clear cache');

    // @todo assert that the site was indeed reverted.
    //   There is however a slight problem, because the test database is not
    //   actually properly populated, something strange is going on.
    // self::assertEquals($original, $this->config('system.site')->get('name'));
    $id = array_key_first($backups);

    $this->drupalGet(sprintf('/admin/config/development/sqlite-backup/delete/%s', $id));
    $this->submitForm([], 'Confirm');

    self::assertCount(0, $this->container->get(SqliteBackupManager::class)->getBackups());

  }

}
