<?php

namespace Drupal\Tests\sqlite_test_cache\Kernel;

use Drupal\KernelTests\KernelTestBase;

/**
 * Base class for sqlite cached kernel tests.
 */
abstract class SqliteCachedKernelTestBase extends KernelTestBase {

  /**
   * Database setup tasks.
   *
   * These are ran only once per whole test suite run.
   */
  protected function setUpDatabase(): void {}

  /**
   * Class setup tasks.
   *
   * These are ran before every test method. Assigning class properties needs
   * to be done here.
   */
  protected function setUpClass(): void {}

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    [$lastAvailableClassCache, $missingClassCaches] = $this->getCacheInfo();

    if ($lastAvailableClassCache) {
      $this->restoreCache($lastAvailableClassCache);
      $this->callSetUpClassOnAncestorsAndSelf($lastAvailableClassCache);
    }

    foreach ($missingClassCaches as $class) {
      $setUpDatabaseMethod = $class->getMethod('setUpDatabase');

      // Only call each setup once. If the parent doesn't implement it
      // themselves don't call the inherited method.
      if ($setUpDatabaseMethod->getDeclaringClass()->getName() == $class->getName()) {
        $class->getMethod('setUpDatabase')->invoke($this);
      }

      $this->saveCache($class);
      $this->callSetUpClassOnAncestorsAndSelf($class, $lastAvailableClassCache);
      $lastAvailableClassCache = $class;
    }

    // CRITICAL: Do NOT clear any caches after restoration!
    // Clearing field definitions causes ContentTranslationHandler to be initialized 
    // with empty field storage definitions, leading to incorrect content_translation_uid 
    // field being added for entities that already have uid fields.
    // The cached state from the database is correct and should be preserved.
  }






  /**
   * Returns the cache status of parent classes.
   */
  protected function getCacheInfo() {
    $lastAvailableClassCache = NULL;
    $missingClassCaches = [];
    $currentClass = new \ReflectionClass($this);
    while ($currentClass->getName() != self::class) {
      $cacheExists = file_exists($this->getDbCachePath($currentClass));
      if (!$cacheExists) {
        $missingClassCaches[] = $currentClass;
      }
      elseif (!$lastAvailableClassCache) {
        $lastAvailableClassCache = $currentClass;
      }
      $currentClass = $currentClass->getParentClass();
    }
    return [$lastAvailableClassCache, array_reverse($missingClassCaches)];
  }

  /**
   * Restores the cached database.
   */
  protected function restoreCache($class) {
    $cachePatch = $this->getDbCachePath($class);
    copy($cachePatch, $this->getDatabasePath());
    
    // Restore KeyValue storage from cache
    $keyValueCachePath = $this->getKeyValueCachePath($class);
    if (file_exists($keyValueCachePath)) {
      $keyValueData = unserialize(file_get_contents($keyValueCachePath));
      if ($keyValueData && $this->container->has('keyvalue')) {
        // Get the KeyValueMemoryFactory and restore its data
        $keyValue = $this->container->get('keyvalue');
        if ($keyValue instanceof \Drupal\Core\KeyValueStore\KeyValueMemoryFactory) {
          $reflection = new \ReflectionObject($keyValue);
          if ($reflection->hasProperty('collections')) {
            $prop = $reflection->getProperty('collections');
            $prop->setAccessible(true);
            
            // Restore each collection
            foreach ($keyValueData as $collectionName => $collectionData) {
              $collection = $keyValue->get($collectionName);
              if ($collection instanceof \Drupal\Core\KeyValueStore\MemoryStorage) {
                $storageReflection = new \ReflectionObject($collection);
                if ($storageReflection->hasProperty('data')) {
                  $dataProp = $storageReflection->getProperty('data');
                  $dataProp->setAccessible(true);
                  $dataProp->setValue($collection, $collectionData);
                }
              }
            }
          }
        }
      }
    }
    
  }

  /**
   * Save the cache.
   */
  protected function saveCache($class) {
    if (!$this->useCache()) {
      return;
    }
    
    $cachePatch = $this->getDbCachePath($class);
    copy($this->getDatabasePath(), $cachePatch);
    
    // Save KeyValue storage to cache
    if ($this->container->has('keyvalue')) {
      $keyValue = $this->container->get('keyvalue');
      if ($keyValue instanceof \Drupal\Core\KeyValueStore\KeyValueMemoryFactory) {
        $reflection = new \ReflectionObject($keyValue);
        if ($reflection->hasProperty('collections')) {
          $prop = $reflection->getProperty('collections');
          $prop->setAccessible(true);
          $collections = $prop->getValue($keyValue);
          
          // Extract data from each collection
          $keyValueData = [];
          foreach ($collections as $collectionName => $collection) {
            if ($collection instanceof \Drupal\Core\KeyValueStore\MemoryStorage) {
              $storageReflection = new \ReflectionObject($collection);
              if ($storageReflection->hasProperty('data')) {
                $dataProp = $storageReflection->getProperty('data');
                $dataProp->setAccessible(true);
                $keyValueData[$collectionName] = $dataProp->getValue($collection);
              }
            }
          }
          
          // Save to file
          $keyValueCachePath = $this->getKeyValueCachePath($class);
          file_put_contents($keyValueCachePath, serialize($keyValueData));
        }
      }
    }
  }

  /**
   * Calls the setUpClass method on the class and its anscestors up to $until.
   */
  protected function callSetUpClassOnAncestorsAndSelf(\ReflectionClass $class, ?\ReflectionClass $until = NULL) {
    $methods = [];

    $lastAnscestors = [self::class];

    if ($until) {
      $lastAnscestors[] = $until->getName();
    }

    while (!in_array($class->getName(), $lastAnscestors)) {
      $method = $class->getMethod('setUpClass');
      while ($method->getDeclaringClass()->getName() != $class->getName()) {
        $class = $class->getParentClass();
      }

      $methods[] = $method;

      $class = $class->getParentClass();
    }

    foreach (array_reverse($methods) as $method) {
      $method->invoke($this);
    }
  }

  /**
   * Returns the path to the database file.
   */
  protected function getDatabasePath() {
    $connection = $this->container->get('database');
    $this->assertInstanceOf('\Drupal\sqlite\Driver\Database\sqlite\Connection', $connection, 'SqliteCachedKernelTestBase can only be used with sqlite database.');
    $options = $connection->getConnectionOptions();
    $prefix = current($connection->getAttachedDatabases());
    return $options['database'] . '-' . $prefix;
  }

  /**
   * Return the patch to cached database for the given class.
   */
  protected function getDbCachePath(\ReflectionClass $class) {
    return dirname($this->getDatabasePath()) . '/cache-' . $class->getShortName();
  }

  /**
   * Return the path to cached KeyValue storage for the given class.
   */
  protected function getKeyValueCachePath(\ReflectionClass $class) {
    return dirname($this->getDatabasePath()) . '/keyvalue-' . $class->getShortName();
  }

  /**
   * Tells if the sqlite test cache should be used.
   */
  protected function useCache() {
    $enabled = TRUE;
    \Drupal::moduleHandler()->alter('sqlite_test_cache_enabled', $enabled);
    return $enabled;
  }




}
