SQL Server Driver for Drupal
=====================

### For Windows or Linux

This contrib module allows the Drupal CMS to connect to Microsoft SQL Server
databases.

Setup
-----

Use [composer](http://getcomposer.org) to install the module:

```bash
composer require drupal/sqlsrv:^5.0
```

With older versions of Drupal, the driver files needed to be copied to the
directory `/drivers` in the webroot of your Drupal installation. This is no
longer necessary. If you have an older version of this driver in that directory
it should be removed.

If you are upgrading from an older version, the database setup must have the
following keys:
```php
    'namespace' => 'Drupal\\sqlsrv\\Driver\\Database\\sqlsrv',
    'autoload' => 'modules/contrib/sqlsrv/src/Driver/Database/sqlsrv',
    // Use the actual filesystem path where the module is installed.
    // Typically: modules/contrib/sqlsrv (Composer install)
    //        or: modules/custom/sqlsrv (manual install)
```

**Quick Summary**:
1. Create Drupal project: `composer create-project drupal/recommended-project mysite`
2. Add this module: `composer require drupal/sqlsrv:^5.0` **BEFORE starting installer**
3. Run Drupal installer and select "SQL Server" as database type

### Regular Expression Support (Optional)

Drupal core allows module developers to use regular expressions within SQL
statements. The base installation does **not** use this feature, so it is **not
required** for Drupal to install.

However, if any contrib modules use regular expressions (REGEXP operator), you will
need to install a **CLR (Common Language Runtime) function** in SQL Server. The driver
expects a function with this signature:

```sql
CREATE FUNCTION [dbo].[REGEXP](
  @pattern NVARCHAR(MAX),
  @matchString NVARCHAR(MAX)
)
RETURNS bit
EXTERNAL NAME [YourAssembly].[YourNamespace.YourClass].[RegexpFunction]
```

**Where:**
- `[dbo]` is your schema name (could be `dbo` or a custom schema)
- `@pattern` is the regular expression pattern
- `@matchString` is the text to match against
- Returns `1` (true) if match found, `0` (false) otherwise
- `[YourAssembly].[YourNamespace.YourClass].[RegexpFunction]` points to your CLR implementation

**Note:** CLR functions require enabling CLR in SQL Server and deploying a .NET assembly.
Most Drupal sites do not need this functionality.

### Minimum Requirements
 * Drupal 10.3 or Drupal 11
 * PHP 8.1 (PHP 8.3 required for Drupal 11)
 * SQL Server 2016
 * pdo_sqlsrv 5.12.0

### Custom Schema Support

The driver supports using a custom database schema instead of the default `dbo` schema. This is useful for:
- Multi-tenant deployments where each Drupal site uses a separate schema
- Security requirements that mandate non-default schemas
- Environments where you don't have permission to use the `dbo` schema

**To configure a custom schema:**

1. **During Installation**: In the database configuration form, expand "Advanced Options" and enter your desired schema name in the "Schema" field.

2. **In settings.php**:
   ```php
   $databases['default']['default'] = [
     'driver' => 'sqlsrv',
     'namespace' => 'Drupal\\sqlsrv\\Driver\\Database\\sqlsrv',
     'autoload' => 'modules/contrib/sqlsrv/src/Driver/Database/sqlsrv/',
     'host' => 'localhost',
     'port' => '1433',
     'database' => 'drupal_db',
     'username' => 'drupal_user',
     'password' => 'your_password',
     'schema' => 'drupal',  // Custom schema name
   ];
   ```

3. **In database URL format**:
   ```php
   $databases['default']['default'] = [
     'database' => 'sqlsrv://user:pass@localhost:1433/drupal_db?schema=drupal&module=sqlsrv&trust_server_certificate=true',
   ];
   ```

**Important Notes:**
- The driver will automatically create the schema during installation if it doesn't exist and the database user has `CREATE SCHEMA` permissions.
- If automatic creation fails, you can manually create the schema before installation:
  ```sql
  CREATE SCHEMA [drupal];
  ```
- Ensure your database user has appropriate permissions on the schema:
  ```sql
  GRANT CREATE TABLE, ALTER, SELECT, INSERT, UPDATE, DELETE ON SCHEMA::[drupal] TO [drupal_user];
  ```

**Alternative Approach:**
Instead of specifying a schema in configuration, you can create a database user with a custom default schema:
```sql
CREATE USER [drupal_user] WITH PASSWORD = 'your_password', DEFAULT_SCHEMA = [drupal];
```

### Connection Options

The driver supports several SQL Server-specific connection options that can be configured in `settings.php` or via database URL parameters.

#### Security and Encryption

**encrypt** (boolean, default: 1)
- Determines whether data should be encrypted before sending over the network.
- Recommended to keep enabled for production environments.
```php
'encrypt' => 1,
```

**trust_server_certificate** (boolean, default: 0)
- When encryption is enabled, allows the server certificate to be trusted even if it can't be verified.
- Useful for self-signed certificates in development environments.
```php
'trust_server_certificate' => 1,
// Or in URL format:
// sqlsrv://user:pass@host/db?module=sqlsrv&trust_server_certificate=true
```

#### High Availability

**multi_subnet_failover** (boolean, default: 0)
- Enable when connecting to an SQL Server availability group listener or failover cluster instance.
- Provides faster detection of and connection to the currently active server.
```php
'multi_subnet_failover' => 1,
```

#### Performance and Behavior

**multiple_active_result_sets** (boolean, default: TRUE)
- Enables Multiple Active Result Sets (MARS).
- Only set to FALSE if you need to disable MARS for specific scenarios.
```php
'multiple_active_result_sets' => FALSE,
```

**transaction_isolation** (integer)
- Sets the transaction isolation level.
- Values: 1 (READ_UNCOMMITTED), 2 (READ_COMMITTED), 4 (REPEATABLE_READ), 8 (SERIALIZABLE), 16 (SNAPSHOT)
```php
'transaction_isolation' => 2,  // READ_COMMITTED
```

**login_timeout** (integer, default: system default)
- Specifies the number of seconds to wait before timing out a login attempt.
```php
'login_timeout' => 30,
```

**pooling** (boolean, default: TRUE)
- Enables connection pooling.
- Only set to FALSE if you need to disable pooling.
```php
'pooling' => FALSE,
```

#### Advanced Features

**column_encryption** (string)
- Enables Always Encrypted functionality for encrypting sensitive data.
- Requires SQL Server 2016+ and proper key configuration.
```php
'column_encryption' => 'Enabled',
```

**key_store_authentication**, **key_store_principal_id**, **key_store_secret**
- Used with Always Encrypted for key management.
- Required when using Azure Key Vault for column encryption keys.
```php
'key_store_authentication' => 'KeyVaultClientSecret',
'key_store_principal_id' => 'your-app-id',
'key_store_secret' => 'your-app-secret',
```

#### Other Options

**appname** (string)
- Sets the application name that appears in SQL Server connection logs.
- Useful for identifying connections from different applications.
```php
'appname' => 'MyDrupalSite',
```

**readonly** (boolean)
- Establishes a read-only connection to the database.
```php
'readonly' => TRUE,
```

**cache_schema** (boolean)
- Controls schema caching behavior.
```php
'cache_schema' => TRUE,
```

**Complete Example:**
```php
$databases['default']['default'] = [
  'driver' => 'sqlsrv',
  'namespace' => 'Drupal\\sqlsrv\\Driver\\Database\\sqlsrv',
  'autoload' => 'modules/contrib/sqlsrv/src/Driver/Database/sqlsrv/',
  'host' => 'localhost',
  'port' => '1433',
  'database' => 'drupal_db',
  'username' => 'drupal_user',
  'password' => 'your_password',
  'schema' => 'drupal',
  'encrypt' => 1,
  'trust_server_certificate' => 1,
  'multi_subnet_failover' => 0,
  'appname' => 'MyDrupalSite',
  'login_timeout' => 30,
];
```

Usage
-----

This driver has a couple peculiarities worth knowing about.

### LIKE expressions

Drupal and the core databases use only two wildcards, `%` and `_`, both of which
are escaped by backslashes. This driver currently uses the default SQL Server
behavior behind-the-scenes, that of escaping the wildcard characters by
enclosing them in brackets `[%]` and `[_]`. When using the `Select::condition()`
function with a LIKE operator, you must use standard Drupal format with
backslash escapes. If you need sqlsrv-specific behavior, you can use
`Select::where()`.
```php
// These two statements are equivalent
$connection->select('test', 't')
  ->condition('t.route', '%[route]%', 'LIKE');
$connection->select('test', 't')
  ->where('t.route LIKE :pattern', [':pattern' => '%[[]route]%']);
```
**Note:** There was a [PDO bug #79276](https://bugs.php.net/bug.php?id=79276) that prevented multiple
`field LIKE :placeholder_x ESCAPE '\'` expressions from appearing in one SQL
statement. This bug has been **fixed in PHP 8.4**. For PHP versions prior to 8.4,
a different escape character must be chosen if you need a custom escape character
multiple times. The driver continues to use bracket escaping for compatibility
with all PHP versions.

### Binary Large Objects

Varbinary data in SQL Server presents two issues. SQL Server is unable to
directly compare a string to a blob without casting the string to varbinary.
This means a query cannot add a ->condition('blob_field', $string) to any
Select query. Thankfully, this is not done in core code, but there is nothing
to stop core from doing so in the future. Contrib modules may currently use
this pattern.

### Non-ASCII strings

Most varchar data is actually stored as nvarchar, because varchar is ascii-only.
Drupal uses UTF-8 while nvarchar encodes data as UCS-2. There are some character
encoding issues that can arise in strange edge cases. Data is typically saved to
varbinary as a stream of UTF-8 characters. If, instead, an nvarchar is converted
into a varbinary, and the binary data extracted into Drupal, it will not be the
same as when it started.

### Collation

This driver supports **both case-insensitive (CI) and case-sensitive (CS)** collations.

**During fresh installation:** If the database doesn't exist and the driver creates it,
the new database will inherit the SQL Server instance's default collation. Most SQL Server
installations default to case-insensitive collations such as `Latin1_General_CI_AS` or
`Latin1_General_100_CI_AS_SC_UTF8`, but this depends on the SQL Server instance configuration.

**During installation checks:** The driver verifies that the database uses a UTF-8
collation with either `_CI_` (case-insensitive) or `_CS_` (case-sensitive) designation.
Both types are fully supported.

**Runtime behavior:** When using case-insensitive collation with the `LIKE BINARY`
operator for case-sensitive matching, the driver automatically adds a `COLLATE` clause
to ensure proper case-sensitive comparison.

### Transactions

The Microsoft ODBC driver, which the PDO driver interfaces with, does
not handle failures within transactions gracefully. If the database produces an
error within a transaction, the entire transaction is automatically rolled back,
even if there are savepoints.

Running Tests
-----

To run the module's kernel tests, you need to configure PHPUnit with a SQL Server connection.

### Local Development

1. Copy `phpunit.xml.dist` to `phpunit.xml`:
   ```bash
   cp phpunit.xml.dist phpunit.xml
   ```

2. Edit `phpunit.xml` and uncomment the `SIMPLETEST_DB` line, then update with your password:
   ```xml
   <env name="SIMPLETEST_DB" value="sqlsrv://sa:YOUR_PASSWORD@sqlsrv/master?module=sqlsrv&amp;trust_server_certificate=true" force="true"/>
   ```

3. The `force="true"` attribute ensures the SQL Server connection overrides any default database configuration (such as DDEV's MySQL default).

4. Run the tests:
   ```bash
   phpunit tests/src/Kernel/
   # Or with DDEV:
   ddev phpunit tests/src/Kernel/
   ```

**Note:** The `phpunit.xml` file is gitignored to prevent accidentally committing credentials.

### CI/CD (GitLab, GitHub Actions, etc.)

For CI/CD pipelines, set `SIMPLETEST_DB` as an environment variable instead of using `phpunit.xml`:

#### GitLab CI Example:
```yaml
test:
  script:
    - export SIMPLETEST_DB="sqlsrv://sa:${SQL_PASSWORD}@sqlsrv/master?module=sqlsrv&trust_server_certificate=true"
    - phpunit tests/src/Kernel/
  variables:
    SQL_PASSWORD: "YourCIPassword"  # Or use GitLab CI/CD masked variables
```

#### GitHub Actions Example:
```yaml
- name: Run tests
  env:
    SIMPLETEST_DB: "sqlsrv://sa:${{ secrets.SQL_PASSWORD }}@sqlsrv/master?module=sqlsrv&trust_server_certificate=true"
  run: phpunit tests/src/Kernel/
```

The `phpunit.xml.dist` file has the `SIMPLETEST_DB` line commented out, so it won't interfere with CI/CD pipelines.

Security Notes
-----

### SA-CORE-2024-008 (Drupal 10.3.9+)

Drupal security advisory [SA-CORE-2024-008](https://www.drupal.org/sa-core-2024-008)
(released November 2024) requires users to allowlist third-party database modules that use
the `Drupal\Core\Database\StatementPrefetch` class.

**This module does NOT use this class**, so **no configuration change is required**.
See [#3488910](https://www.drupal.org/project/sqlsrv/issues/3488910) for details.

Support and Issue Reporting
-----

### Test Status

The 5.0.x branch passes all core Drupal database tests and has been validated for
production use with Drupal 10.3+ and Drupal 11.

### Known Limitations

The driver has a few inherent limitations due to SQL Server and ODBC driver differences
from MySQL/PostgreSQL (documented above):

- **LIKE expressions**: Uses bracket escaping instead of backslash escaping
- **Binary Large Objects**: Cannot directly compare strings to varbinary without CAST
- **Transactions**: ODBC driver automatically rolls back on errors (cannot be fixed in PHP)

These are documented SQL Server behaviors, not bugs in the driver.

### Reporting Issues

If you encounter bugs or have feature requests:
1. Check the [project issue queue](https://www.drupal.org/project/issues/sqlsrv) for existing reports
2. Create a new issue with:
   - Drupal version
   - SQL Server version
   - PHP and pdo_sqlsrv extension versions
   - Detailed steps to reproduce the problem
   - Any error messages from logs
