<?php

declare(strict_types=1);

namespace Drupal\sqlsrv\Driver\Database\sqlsrv;

use Drupal\Core\Database\Transaction\ClientConnectionTransactionState;
use Drupal\Core\Database\Transaction\TransactionManagerBase;

/**
 * SQL Server implementation of TransactionManagerInterface.
 *
 * SQL Server has specific transaction and savepoint syntax that differs
 * from the ANSI SQL standard:
 * - Uses SAVE TRANSACTION instead of SAVEPOINT.
 * - Uses ROLLBACK TRANSACTION instead of ROLLBACK TO SAVEPOINT.
 * - Does not support RELEASE SAVEPOINT.
 *
 * Additionally, SQL Server's ODBC driver has a behavior where if an error
 * occurs within a transaction, the entire transaction is automatically
 * rolled back, even if savepoints exist. This is handled by catching
 * exceptions during savepoint operations.
 */
class TransactionManager extends TransactionManagerBase {

  /**
   * {@inheritdoc}
   */
  protected function beginClientTransaction(): bool {
    return $this->connection->getClientConnection()->beginTransaction();
  }

  /**
   * {@inheritdoc}
   *
   * SQL Server uses SAVE TRANSACTION syntax instead of SAVEPOINT.
   */
  protected function addClientSavepoint(string $name): bool {
    /** @var \Drupal\sqlsrv\Driver\Database\sqlsrv\Connection $connection */
    $connection = $this->connection;
    $connection->queryDirect('SAVE TRANSACTION ' . $name);
    return TRUE;
  }

  /**
   * {@inheritdoc}
   *
   * SQL Server uses ROLLBACK TRANSACTION syntax instead of ROLLBACK TO
   * SAVEPOINT.
   */
  protected function rollbackClientSavepoint(string $name): bool {
    try {
      /** @var \Drupal\sqlsrv\Driver\Database\sqlsrv\Connection $connection */
      $connection = $this->connection;
      $connection->queryDirect('ROLLBACK TRANSACTION ' . $name);
    }
    catch (\Exception $e) {
      // If an exception occurred within a savepoint, the savepoint may be
      // automatically destroyed in SQL Server's ODBC driver. We want to know
      // the real cause for the exception occurring, so catch and ignore any
      // "savepoint not found" exception, to allow any try/catch block further
      // up the stack to throw their exception.
      if (strpos($e->getMessage(), 'No transaction or savepoint of that name was found.') !== FALSE) {
        // Savepoint was already rolled back, ignore the error.
        return TRUE;
      }
      throw $e;
    }
    return TRUE;
  }

  /**
   * {@inheritdoc}
   *
   * SQL Server does not support RELEASE SAVEPOINT.
   * Savepoints are automatically released when the transaction commits or
   * when a later savepoint is created.
   */
  protected function releaseClientSavepoint(string $name): bool {
    // SQL Server does not support RELEASE SAVEPOINT.
    // Savepoints are implicitly released, so this is a no-op.
    return TRUE;
  }

  /**
   * {@inheritdoc}
   */
  protected function commitClientTransaction(): bool {
    $clientCommit = $this->connection->getClientConnection()->commit();
    $this->setConnectionTransactionState($clientCommit ?
      ClientConnectionTransactionState::Committed :
      ClientConnectionTransactionState::CommitFailed
    );
    return $clientCommit;
  }

  /**
   * {@inheritdoc}
   */
  protected function rollbackClientTransaction(): bool {
    $clientRollback = $this->connection->getClientConnection()->rollBack();
    $this->setConnectionTransactionState($clientRollback ?
      ClientConnectionTransactionState::RolledBack :
      ClientConnectionTransactionState::RollbackFailed
    );
    return $clientRollback;
  }

}
