<?php

namespace Drupal\Tests\sqlsrv\Kernel;

use Drupal\Core\Database\Database;

/**
 * Tests collation handling for case-sensitive and case-insensitive databases.
 *
 * SQL Server can use either case-sensitive (_CS_) or case-insensitive (_CI_)
 * collations. This test validates that the driver works correctly with both.
 *
 * @group Database
 */
class CollationHandlingTest extends SqlsrvTestBase {

  /**
   * Tests that isCaseSensitive() correctly detects collation.
   */
  public function testIsCaseSensitive() {
    $connection = Database::getConnection();
    /** @var \Drupal\sqlsrv\Driver\Database\sqlsrv\Schema $schema */
    $schema = $connection->schema();

    $collation = $schema->getCollation();
    $is_case_sensitive = $schema->isCaseSensitive();

    // Verify the result matches the collation string.
    if (stripos($collation, '_CS_') !== FALSE) {
      $this->assertTrue($is_case_sensitive, 'Database with _CS_ collation should be detected as case-sensitive.');
    }
    elseif (stripos($collation, '_CI_') !== FALSE) {
      $this->assertFalse($is_case_sensitive, 'Database with _CI_ collation should be detected as case-insensitive.');
    }
    else {
      $this->fail("Unknown collation format: {$collation}");
    }
  }

  /**
   * Tests LIKE operator behavior with current collation.
   */
  public function testLikeOperator() {
    $connection = Database::getConnection();
    /** @var \Drupal\sqlsrv\Driver\Database\sqlsrv\Schema $schema */
    $schema = $connection->schema();

    // Create a test table.
    $table_spec = [
      'fields' => [
        'id' => [
          'type' => 'serial',
          'not null' => TRUE,
        ],
        'name' => [
          'type' => 'varchar',
          'length' => 255,
          'not null' => TRUE,
        ],
      ],
      'primary key' => ['id'],
    ];

    $schema->createTable('test_like_collation', $table_spec);

    try {
      // Insert test data.
      $connection->insert('test_like_collation')
        ->fields(['name'], ['TestValue'])
        ->execute();

      // Test regular LIKE - behavior depends on database collation.
      $result = $connection->select('test_like_collation', 't')
        ->fields('t', ['name'])
        ->condition('name', 'testvalue', 'LIKE')
        ->execute()
        ->fetchField();

      $is_case_sensitive = $schema->isCaseSensitive();

      if ($is_case_sensitive) {
        // Case-sensitive: 'testvalue' should NOT match 'TestValue'.
        $this->assertFalse($result, 'On case-sensitive database, LIKE should be case-sensitive.');
      }
      else {
        // Case-insensitive: 'testvalue' SHOULD match 'TestValue'.
        $this->assertEquals('TestValue', $result, 'On case-insensitive database, LIKE should be case-insensitive.');
      }
    }
    finally {
      $schema->dropTable('test_like_collation');
    }
  }

  /**
   * Tests LIKE BINARY operator for case-sensitive matching.
   */
  public function testLikeBinaryOperator() {
    $connection = Database::getConnection();
    /** @var \Drupal\sqlsrv\Driver\Database\sqlsrv\Schema $schema */
    $schema = $connection->schema();

    // Create a test table.
    $table_spec = [
      'fields' => [
        'id' => [
          'type' => 'serial',
          'not null' => TRUE,
        ],
        'name' => [
          'type' => 'varchar',
          'length' => 255,
          'not null' => TRUE,
        ],
      ],
      'primary key' => ['id'],
    ];

    $schema->createTable('test_like_binary', $table_spec);

    try {
      // Insert test data with different cases.
      $connection->insert('test_like_binary')
        ->fields(['name'])
        ->values(['TestValue'])
        ->values(['testvalue'])
        ->values(['TESTVALUE'])
        ->execute();

      // Test LIKE BINARY - should always be case-sensitive.
      // Match exact case only.
      $result = $connection->select('test_like_binary', 't')
        ->fields('t', ['name'])
        ->condition('name', 'TestValue', 'LIKE BINARY')
        ->execute()
        ->fetchAll();

      // LIKE BINARY should match only exact case.
      $this->assertCount(1, $result, 'LIKE BINARY should match exactly one record.');
      $this->assertEquals('TestValue', $result[0]->name, 'LIKE BINARY should match exact case only.');

      // Test that lowercase doesn't match.
      $result2 = $connection->select('test_like_binary', 't')
        ->fields('t', ['name'])
        ->condition('name', 'testvalue', 'LIKE BINARY')
        ->execute()
        ->fetchAll();

      $this->assertCount(1, $result2, 'LIKE BINARY should match exactly one record.');
      $this->assertEquals('testvalue', $result2[0]->name, 'LIKE BINARY should match exact case only.');
    }
    finally {
      $schema->dropTable('test_like_binary');
    }
  }

  /**
   * Tests that binary field spec creates case-sensitive collation.
   */
  public function testBinaryFieldSpec() {
    $connection = Database::getConnection();
    /** @var \Drupal\sqlsrv\Driver\Database\sqlsrv\Schema $schema */
    $schema = $connection->schema();

    // Only test on case-insensitive databases.
    if ($schema->isCaseSensitive()) {
      $this->markTestSkipped('This test only applies to case-insensitive databases.');
    }

    // Create a table with a binary (case-sensitive) text field.
    $table_spec = [
      'fields' => [
        'id' => [
          'type' => 'serial',
          'not null' => TRUE,
        ],
        'cs_field' => [
          'type' => 'varchar',
          'length' => 255,
          'not null' => TRUE,
          'binary' => TRUE,
        ],
      ],
      'primary key' => ['id'],
    ];

    $schema->createTable('test_binary_field', $table_spec);

    try {
      // Check that the column was created with case-sensitive collation.
      $column_collation = $schema->getCollation('test_binary_field', 'cs_field');

      // The column should have _CS_ in its collation.
      $this->assertStringContainsString('_CS_', $column_collation,
        'Field with binary=TRUE should use case-sensitive collation.');

      // Insert and test data.
      $connection->insert('test_binary_field')
        ->fields(['cs_field'])
        ->values(['TestValue'])
        ->execute();

      // Query with different case - should NOT match on CS column.
      $result = $connection->select('test_binary_field', 't')
        ->fields('t', ['cs_field'])
        ->condition('cs_field', 'testvalue')
        ->execute()
        ->fetchField();

      $this->assertFalse($result, 'Case-sensitive column should not match different case.');

      // Query with exact case - SHOULD match.
      $result2 = $connection->select('test_binary_field', 't')
        ->fields('t', ['cs_field'])
        ->condition('cs_field', 'TestValue')
        ->execute()
        ->fetchField();

      $this->assertEquals('TestValue', $result2, 'Case-sensitive column should match exact case.');
    }
    finally {
      $schema->dropTable('test_binary_field');
    }
  }

}
