<?php

namespace Drupal\Tests\sqlsrv\Kernel;

use Drupal\Core\Database\TransactionNameNonUniqueException;
use Drupal\Core\Database\TransactionOutOfOrderException;

/**
 * Tests transaction handling with SQL Server TransactionManager.
 *
 * @group Database
 */
class TransactionTest extends SqlsrvTestBase {

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    // Skip tests if pdo_sqlsrv extension is not available.
    if (!extension_loaded('pdo_sqlsrv')) {
      // Set siteDirectory to prevent tearDown errors.
      $this->siteDirectory = 'vfs://';
      $this->markTestSkipped('The pdo_sqlsrv PHP extension is not available.');
      return;
    }
    parent::setUp();

    // Create a test table for transaction testing.
    $this->connection->schema()->createTable('test_transaction', [
      'fields' => [
        'id' => [
          'type' => 'serial',
          'not null' => TRUE,
        ],
        'name' => [
          'type' => 'varchar',
          'length' => 255,
          'not null' => TRUE,
        ],
      ],
      'primary key' => ['id'],
    ]);
  }

  /**
   * Tests basic transaction commit.
   *
   * @covers \Drupal\sqlsrv\Driver\Database\sqlsrv\TransactionManager::beginClientTransaction
   * @covers \Drupal\sqlsrv\Driver\Database\sqlsrv\TransactionManager::commitClientTransaction
   */
  public function testBasicCommit(): void {
    $txn = $this->connection->startTransaction();

    // Insert data within transaction.
    $this->connection->insert('test_transaction')
      ->fields(['name' => 'test1'])
      ->execute();

    // Data should exist before commit.
    $count = $this->connection->select('test_transaction')
      ->countQuery()
      ->execute()
      ->fetchField();
    $this->assertEquals(1, $count, 'Data exists within transaction.');

    // Commit by destroying transaction object.
    unset($txn);

    // Data should persist after commit.
    $count = $this->connection->select('test_transaction')
      ->countQuery()
      ->execute()
      ->fetchField();
    $this->assertEquals(1, $count, 'Data persists after commit.');
  }

  /**
   * Tests basic transaction rollback.
   *
   * @covers \Drupal\sqlsrv\Driver\Database\sqlsrv\TransactionManager::beginClientTransaction
   * @covers \Drupal\sqlsrv\Driver\Database\sqlsrv\TransactionManager::rollbackClientTransaction
   */
  public function testBasicRollback(): void {
    $txn = $this->connection->startTransaction();

    // Insert data within transaction.
    $this->connection->insert('test_transaction')
      ->fields(['name' => 'test2'])
      ->execute();

    // Data should exist before rollback.
    $count = $this->connection->select('test_transaction')
      ->countQuery()
      ->execute()
      ->fetchField();
    $this->assertEquals(1, $count, 'Data exists within transaction.');

    // Explicitly rollback.
    $txn->rollback();

    // Data should not persist after rollback.
    $count = $this->connection->select('test_transaction')
      ->countQuery()
      ->execute()
      ->fetchField();
    $this->assertEquals(0, $count, 'Data does not persist after rollback.');
  }

  /**
   * Tests savepoint creation and commit.
   *
   * @covers \Drupal\sqlsrv\Driver\Database\sqlsrv\TransactionManager::addClientSavepoint
   * @covers \Drupal\sqlsrv\Driver\Database\sqlsrv\TransactionManager::releaseClientSavepoint
   */
  public function testSavepointCommit(): void {
    $txn1 = $this->connection->startTransaction();

    $this->connection->insert('test_transaction')
      ->fields(['name' => 'outer'])
      ->execute();

    // Create a savepoint.
    $txn2 = $this->connection->startTransaction();

    $this->connection->insert('test_transaction')
      ->fields(['name' => 'inner'])
      ->execute();

    // Both records should exist.
    $count = $this->connection->select('test_transaction')
      ->countQuery()
      ->execute()
      ->fetchField();
    $this->assertEquals(2, $count, 'Both records exist within nested transactions.');

    // Commit inner transaction (release savepoint).
    unset($txn2);

    // Commit outer transaction.
    unset($txn1);

    // Both records should persist.
    $count = $this->connection->select('test_transaction')
      ->countQuery()
      ->execute()
      ->fetchField();
    $this->assertEquals(2, $count, 'Both records persist after commit.');
  }

  /**
   * Tests savepoint rollback.
   *
   * @covers \Drupal\sqlsrv\Driver\Database\sqlsrv\TransactionManager::addClientSavepoint
   * @covers \Drupal\sqlsrv\Driver\Database\sqlsrv\TransactionManager::rollbackClientSavepoint
   */
  public function testSavepointRollback(): void {
    $txn1 = $this->connection->startTransaction();

    $this->connection->insert('test_transaction')
      ->fields(['name' => 'outer'])
      ->execute();

    // Create a savepoint.
    $txn2 = $this->connection->startTransaction();

    $this->connection->insert('test_transaction')
      ->fields(['name' => 'inner'])
      ->execute();

    // Rollback inner transaction (rollback to savepoint).
    $txn2->rollback();

    // Only outer record should exist.
    $count = $this->connection->select('test_transaction')
      ->countQuery()
      ->execute()
      ->fetchField();
    $this->assertEquals(1, $count, 'Only outer record exists after savepoint rollback.');

    // Commit outer transaction.
    unset($txn1);

    // Only outer record should persist.
    $count = $this->connection->select('test_transaction')
      ->countQuery()
      ->execute()
      ->fetchField();
    $this->assertEquals(1, $count, 'Only outer record persists after commit.');

    $name = $this->connection->select('test_transaction', 't')
      ->fields('t', ['name'])
      ->execute()
      ->fetchField();
    $this->assertEquals('outer', $name, 'Correct record persists.');
  }

  /**
   * Tests multiple nested savepoints.
   *
   * @covers \Drupal\sqlsrv\Driver\Database\sqlsrv\TransactionManager::addClientSavepoint
   * @covers \Drupal\sqlsrv\Driver\Database\sqlsrv\TransactionManager::rollbackClientSavepoint
   */
  public function testMultipleSavepoints(): void {
    $txn1 = $this->connection->startTransaction();
    $this->connection->insert('test_transaction')
      ->fields(['name' => 'level1'])
      ->execute();

    $txn2 = $this->connection->startTransaction();
    $this->connection->insert('test_transaction')
      ->fields(['name' => 'level2'])
      ->execute();

    $txn3 = $this->connection->startTransaction();
    $this->connection->insert('test_transaction')
      ->fields(['name' => 'level3'])
      ->execute();

    // All three records should exist.
    $count = $this->connection->select('test_transaction')
      ->countQuery()
      ->execute()
      ->fetchField();
    $this->assertEquals(3, $count, 'All three records exist.');

    // Rollback level 3.
    $txn3->rollback();

    $count = $this->connection->select('test_transaction')
      ->countQuery()
      ->execute()
      ->fetchField();
    $this->assertEquals(2, $count, 'Only two records after level3 rollback.');

    // Commit level 2.
    unset($txn2);

    // Commit level 1.
    unset($txn1);

    // Two records should persist.
    $count = $this->connection->select('test_transaction')
      ->countQuery()
      ->execute()
      ->fetchField();
    $this->assertEquals(2, $count, 'Two records persist after commit.');
  }

  /**
   * Tests transaction rollback out of order.
   *
   * @covers \Drupal\sqlsrv\Driver\Database\sqlsrv\TransactionManager::rollbackClientSavepoint
   */
  public function testRollbackOutOfOrder(): void {
    $txn1 = $this->connection->startTransaction('txn1');
    $txn2 = $this->connection->startTransaction('txn2');
    $txn3 = $this->connection->startTransaction('txn3');
    $this->expectException(TransactionOutOfOrderException::class);
    $txn2->rollback();
  }

  /**
   * Tests rolling back an already rolled back transaction.
   *
   * @covers \Drupal\sqlsrv\Driver\Database\sqlsrv\TransactionManager::rollbackClientSavepoint
   */
  public function testRollbackTransactionName(): void {
    $txn1 = $this->connection->startTransaction('txn1');
    $txn2 = $this->connection->startTransaction('txn2');
    $txn2->rollback();
    // Rolling back an already-rolled-back transaction should throw an
    // exception. This can be either TransactionNoActiveException
    // (transaction not in stack) or TransactionOutOfOrderException
    // (transaction can't be found in order check). Both are valid error
    // conditions for this scenario.
    $this->expectException(\Exception::class);
    $this->expectExceptionMessageMatches('/transaction|txn2/i');
    // Try to roll it back again.
    $txn2->rollback();
  }

  /**
   * Tests transaction must have unique names.
   *
   * @covers \Drupal\sqlsrv\Driver\Database\sqlsrv\Connection::pushTransaction
   */
  public function testDuplicateTransactionName(): void {
    $txn1 = $this->connection->startTransaction('txn1');
    $txn2 = $this->connection->startTransaction('txn2');
    $this->expectException(TransactionNameNonUniqueException::class);
    $txn3 = $this->connection->startTransaction('txn2');
  }

  /**
   * Tests complex transaction scenario with updates and deletes.
   *
   * @covers \Drupal\sqlsrv\Driver\Database\sqlsrv\TransactionManager
   */
  public function testComplexTransactionScenario(): void {
    // Insert initial data.
    $this->connection->insert('test_transaction')
      ->fields(['name' => 'initial'])
      ->execute();

    $txn1 = $this->connection->startTransaction();

    // Update within transaction.
    $this->connection->update('test_transaction')
      ->fields(['name' => 'updated'])
      ->condition('name', 'initial')
      ->execute();

    // Insert additional record.
    $this->connection->insert('test_transaction')
      ->fields(['name' => 'additional'])
      ->execute();

    $txn2 = $this->connection->startTransaction();

    // Delete within savepoint.
    $this->connection->delete('test_transaction')
      ->condition('name', 'additional')
      ->execute();

    // One record should remain.
    $count = $this->connection->select('test_transaction')
      ->countQuery()
      ->execute()
      ->fetchField();
    $this->assertEquals(1, $count, 'One record remains after delete in savepoint.');

    // Rollback savepoint - delete should be undone.
    $txn2->rollback();

    $count = $this->connection->select('test_transaction')
      ->countQuery()
      ->execute()
      ->fetchField();
    $this->assertEquals(2, $count, 'Two records exist after savepoint rollback.');

    // Commit outer transaction.
    unset($txn1);

    // Verify final state.
    $names = $this->connection->select('test_transaction', 't')
      ->fields('t', ['name'])
      ->execute()
      ->fetchCol();
    sort($names);
    $this->assertEquals(['additional', 'updated'], $names, 'Final state is correct.');
  }

}
