<?php

namespace Drupal\sqlsrv\Driver\Database\sqlsrv\Install;

use Drupal\Core\Database\Database;
use Drupal\Core\Database\Install\Tasks as InstallTasks;
use Drupal\Core\Database\DatabaseNotFoundException;
use Drupal\sqlsrv\Driver\Database\sqlsrv\Connection;
use Drupal\sqlsrv\Driver\Database\sqlsrv\Utils;

/**
 * Specifies installation tasks for PostgreSQL databases.
 */
class Tasks extends InstallTasks {

  /**
   * {@inheritdoc}
   */
  protected $pdoDriver = 'sqlsrv';

  /**
   * Constructs a \Drupal\Core\Database\Driver\pgsql\Install\Tasks object.
   */
  public function __construct() {
    $this->tasks[] = [
      'function' => 'checkDatabaseVersion',
      'arguments' => [],
    ];
    $this->tasks[] = [
      'function' => 'checkEncoding',
      'arguments' => [],
    ];
    $this->tasks[] = [
      'function' => 'ensureSchemaExists',
      'arguments' => [],
    ];
    $this->tasks[] = [
      'function' => 'initializeDatabase',
      'arguments' => [],
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function name() {
    return t('SQLServer');
  }

  /**
   * {@inheritdoc}
   */
  public function minimumVersion() {
    // SQL Server 2019 - 15.x - 2025-01-07.
    // SQL Server 2017 - 14.x - 2022-10-11.
    // SQL Server 2016 - 13.x - 2021-07-13.
    return '13.0';
  }

  /**
   * Check if the database version meets minimum requirements.
   *
   * Overridden to handle Azure SQL Database versioning, which differs from
   * SQL Server versioning. Azure SQL Database is always kept up to date and
   * uses different version numbers than on-premise SQL Server.
   *
   * Azure SQL Database v12 is roughly equivalent to SQL Server 2016/2017,
   * and Azure is automatically upgraded, so we skip version checks for Azure.
   */
  protected function checkDatabaseVersion() {
    try {
      $connection = Database::getConnection();

      // Check if this is Azure SQL Database.
      $edition_result = $connection->query("SELECT SERVERPROPERTY('Edition') AS Edition")->fetchObject();
      $edition = $edition_result->Edition ?? '';

      // Azure SQL Database editions include "SQL Azure" or "SQL Database".
      // These are always up-to-date, so skip version check.
      if (stripos($edition, 'SQL Azure') !== FALSE || stripos($edition, 'Azure') !== FALSE) {
        $this->pass(t('Azure SQL detected (Edition: @edition). Version check skipped (Azure is always up-to-date).', [
          '@edition' => $edition,
        ]));
        return;
      }

      // For non-Azure SQL Server, perform standard version check.
      if ($this->minimumVersion() && version_compare($connection->version(), $this->minimumVersion() . '-AnyName', '<')) {
        $this->fail(t("The database server version %version is less than the minimum required version %minimum_version.", [
          '%version' => $connection->version(),
          '%minimum_version' => $this->minimumVersion(),
        ]));
      }
    }
    catch (\Exception $e) {
      $this->fail(t('Failed to check database version: @error', ['@error' => $e->getMessage()]));
    }
  }

  /**
   * {@inheritdoc}
   */
  protected function connect() {
    try {
      // This doesn't actually test the connection.
      Database::setActiveConnection();
      // Now actually do a check.
      Database::getConnection();
      $this->pass('Drupal can CONNECT to the database ok.');
    }
    catch (\Exception $e) {
      // Attempt to create the database if it is not found.
      if ($e->getCode() == Connection::DATABASE_NOT_FOUND) {
        // Remove the database string from connection info.
        $connection_info = Database::getConnectionInfo();
        $database = $connection_info['default']['database'];
        unset($connection_info['default']['database']);

        // In order to change the Database::$databaseInfo array, need to remove
        // the active connection, then re-add it with the new info.
        Database::removeConnection('default');
        Database::addConnectionInfo('default', 'default', $connection_info['default']);

        try {
          // Now, attempt the connection again; if it's successful, attempt to
          // create the database.
          Database::getConnection()->createDatabase($database);
          Database::closeConnection();

          // Now, restore the database config.
          Database::removeConnection('default');
          $connection_info['default']['database'] = $database;
          Database::addConnectionInfo('default', 'default', $connection_info['default']);

          // Check the database connection.
          Database::getConnection();
          $this->pass('Drupal can CONNECT to the database ok.');
        }
        catch (DatabaseNotFoundException $e) {
          // Still no dice; probably a permission issue. Raise the error to the
          // installer.
          $data = ['%database' => $database, '%error' => $e->getMessage()];
          $this->fail(t('Database %database not found. The server reports the following message when attempting to create the database: %error.', $data));
          return FALSE;
        }
      }
      else {
        // Database connection failed for some other reason than the database
        // not existing.
        $this->fail(t('Failed to connect to your database server. The server reports the following message: %error.<ul><li>Is the database server running?</li><li>Does the database exist, and have you entered the correct database name?</li><li>Have you entered the correct username and password?</li><li>Have you entered the correct database hostname?</li></ul>', ['%error' => $e->getMessage()]));
        return FALSE;
      }
    }
    return TRUE;
  }

  /**
   * Check encoding is UTF8 and collation.
   */
  protected function checkEncoding() {
    try {
      $database = Database::getConnection();

      /** @var \Drupal\sqlsrv\Driver\Database\sqlsrv\Schema $schema */
      $schema = $database->schema();
      $collation = $schema->getCollation();

      // Check if collation is case-sensitive or case-insensitive.
      // Both are now supported.
      $is_case_sensitive = stripos($collation, '_CS_') !== FALSE;
      $is_case_insensitive = stripos($collation, '_CI_') !== FALSE;

      if ($is_case_insensitive) {
        $this->pass(t('Database is using case-insensitive collation: @collation', [
          '@collation' => $collation,
        ]));
      }
      elseif ($is_case_sensitive) {
        $this->pass(t('Database is using case-sensitive collation: @collation', [
          '@collation' => $collation,
        ]));
      }
      else {
        // Neither CI nor CS detected - unusual collation.
        $data = [
          '%current' => $collation,
          '%driver' => $this->name(),
        ];
        $this->fail(t('The %driver database is using %current collation. Use a standard SQL Server collation with either _CI_ (case-insensitive) or _CS_ (case-sensitive) designation.', $data));
      }

      // Check for UTF-8 encoding.
      if (stripos($collation, 'UTF8') === FALSE && stripos($collation, 'UTF-8') === FALSE) {
        $this->fail(t('Database collation @collation does not appear to be UTF-8. Use a UTF-8 collation.', [
          '@collation' => $collation,
        ]));
      }
    }
    catch (\Exception $e) {
      $this->fail(t('Failed to check database collation: @error', [
        '@error' => $e->getMessage(),
      ]));
    }
  }

  /**
   * Ensure the configured schema exists, creating it if necessary.
   */
  protected function ensureSchemaExists() {
    try {
      /** @var \Drupal\sqlsrv\Driver\Database\sqlsrv\Connection $connection */
      $connection = Database::getConnection();
      /** @var \Drupal\sqlsrv\Driver\Database\sqlsrv\Schema $schema */
      $schema = $connection->schema();

      $connection_info = Database::getConnectionInfo();
      $configured_schema = $connection_info['default']['schema'] ?? NULL;

      // If a custom schema is configured, ensure it exists.
      if (!empty($configured_schema)) {
        if (!$schema->schemaExists($configured_schema)) {
          // Schema doesn't exist, try to create it.
          try {
            $schema->createSchema($configured_schema);
            $this->pass(t('Created database schema: @schema', ['@schema' => $configured_schema]));
          }
          catch (\Exception $e) {
            $this->fail(t('Failed to create database schema @schema: @error. Create the schema manually or ensure the database user has CREATE SCHEMA permissions.', [
              '@schema' => $configured_schema,
              '@error' => $e->getMessage(),
            ]));
            return FALSE;
          }
        }
        else {
          $this->pass(t('Database schema @schema exists.', ['@schema' => $configured_schema]));
        }
      }
      else {
        // No custom schema configured, use the default schema.
        $default_schema = $schema->getDefaultSchema();
        $this->pass(t('Using default database schema: @schema', ['@schema' => $default_schema]));
      }
    }
    catch (\Exception $e) {
      $this->fail(t('Failed to verify database schema: @error', ['@error' => $e->getMessage()]));
      return FALSE;
    }
    return TRUE;
  }

  /**
   * Make SQLServer Drupal friendly.
   */
  public function initializeDatabase() {
    // We create some functions using global names instead of prefixing them
    // like we do with table names. This is so that we don't double up if more
    // than one instance of Drupal is running on a single database. We therefore
    // avoid trying to create them again in that case.
    try {

      /** @var \Drupal\sqlsrv\Driver\Database\sqlsrv\Connection $connection */
      $connection = Database::getConnection();

      Utils::DeployCustomFunctions($connection);

      $this->pass(t('SQLServer has initialized itself.'));
    }
    catch (\Exception $e) {
      $this->fail(t('Drupal could not be correctly setup with the existing database. Revise any errors.'));
    }
  }

  /**
   * {@inheritdoc}
   */
  public function getFormOptions(array $database) {
    $form = parent::getFormOptions($database);
    if (empty($form['advanced_options']['port']['#default_value'])) {
      $form['advanced_options']['port']['#default_value'] = '1433';
    }
    $form['advanced_options']['schema'] = [
      '#type' => 'textfield',
      '#title' => t('Schema'),
      '#default_value' => empty($database['schema']) ? 'dbo' : $database['schema'],
      '#size' => 10,
      '#required' => FALSE,
    ];
    $form['advanced_options']['cache_schema'] = [
      '#type' => 'checkbox',
      '#title' => t('Cache Schema Definitions'),
      '#description' => t('Allow the table schema to be cached. This will significantly speed up the site, but the schema must be stable.'),
      '#return_value' => 'true',
    ];
    $form['advanced_options']['autoload'] = [
      '#type' => 'textfield',
      '#title' => t('Driver Path'),
      '#description' => t('File path - will be automatically filled in when written to the settings file.'),
      '#default_value' => '',
    ];
    $form['advanced_options']['encrypt'] = [
      '#type' => 'checkbox',
      '#title' => t('Encrypt'),
      '#description' => t('Determines whether data should be encrypted before sending over the network. It is recommended to keep this enabled.'),
      '#default_value' => '1',
      '#return_value' => '1',
    ];
    $form['advanced_options']['trust_server_certificate'] = [
      '#type' => 'checkbox',
      '#title' => t('Trust Server Certificate'),
      '#description' => t("When encryption is enabled, allows the server certificate to be trusted even if it can't be verified, for example when the server uses a self-signed certificate."),
      '#default_value' => !empty($database['trust_server_certificate']) ? '1' : FALSE,
      '#return_value' => '1',
    ];
    $form['advanced_options']['multi_subnet_failover'] = [
      '#type' => 'checkbox',
      '#title' => t('Multi Subnet Failover'),
      '#description' => t('If connecting to an SQL Server availability group listener or failover cluster instance, enable this option to provide faster detection of and connection to the currently active server.'),
      '#default_value' => FALSE,
      '#return_value' => '1',
    ];

    // Make username not required.
    $form['username']['#required'] = FALSE;
    // Add a description for about leaving username blank.
    $form['username']['#description'] = t('Leave username (and password) blank to use Windows authentication.');
    return $form;
  }

}
