<?php

namespace Drupal\Tests\sqlsrv\Kernel;

use Drupal\Core\Database\Database;

/**
 * Tests schema caching functionality.
 *
 * Validates that when cache_schema is enabled:
 * - Column information is cached in memory
 * - Subsequent queries use the cache instead of querying the database
 * - Cache can be cleared.
 *
 * @group Database
 */
class SchemaCacheTest extends SqlsrvTestBase {

  /**
   * Tests that schema caching works correctly.
   */
  public function testSchemaCaching() {
    $connection = Database::getConnection();
    /** @var \Drupal\sqlsrv\Driver\Database\sqlsrv\Schema $schema */
    $schema = $connection->schema();

    // Get connection options to check cache_schema setting.
    $connection_options = $connection->getConnectionOptions();
    $cache_enabled = !empty($connection_options['cache_schema']);

    // Create a test table.
    $table_spec = [
      'fields' => [
        'id' => [
          'type' => 'serial',
          'not null' => TRUE,
        ],
        'name' => [
          'type' => 'varchar',
          'length' => 255,
          'not null' => TRUE,
        ],
        'data' => [
          'type' => 'blob',
          'size' => 'big',
          'not null' => FALSE,
        ],
      ],
      'primary key' => ['id'],
    ];

    $schema->createTable('test_schema_cache', $table_spec);

    try {
      // Query column information for the first time.
      $info1 = $schema->queryColumnInformation('test_schema_cache');

      // Verify the structure is correct.
      $this->assertIsArray($info1);
      $this->assertArrayHasKey('columns', $info1);
      $this->assertArrayHasKey('blobs', $info1);
      $this->assertArrayHasKey('id', $info1['columns']);
      $this->assertArrayHasKey('name', $info1['columns']);
      $this->assertArrayHasKey('data', $info1['columns']);

      // Blob column should be in the blobs array.
      $this->assertArrayHasKey('data', $info1['blobs']);

      // Query again - should use cache if enabled.
      $info2 = $schema->queryColumnInformation('test_schema_cache');

      // Results should be identical.
      $this->assertEquals($info1, $info2);

      // To verify caching is working, we can check that the arrays are
      // actually the same instance in memory (reference equality) when
      // caching is enabled. This is implementation-specific but demonstrates
      // the cache is being used.
      if ($cache_enabled) {
        // When cached, subsequent calls return the exact same array reference.
        // We can't directly test reference equality, but we can verify that
        // modifying the cached data affects the next call.
        // However, this would be testing implementation details.
        // Instead, we verify that calling resetColumnInformation clears cache.
        $schema->resetColumnInformation('test_schema_cache');

        // After reset, querying again should work.
        $info3 = $schema->queryColumnInformation('test_schema_cache');
        $this->assertEquals($info1, $info3);
      }
    }
    finally {
      $schema->dropTable('test_schema_cache');
    }
  }

  /**
   * Tests that column information includes all expected data.
   */
  public function testColumnInformationStructure() {
    $connection = Database::getConnection();
    /** @var \Drupal\sqlsrv\Driver\Database\sqlsrv\Schema $schema */
    $schema = $connection->schema();

    // Create a table with various column types.
    $table_spec = [
      'fields' => [
        'id' => [
          'type' => 'serial',
          'not null' => TRUE,
        ],
        'name' => [
          'type' => 'varchar',
          'length' => 255,
          'not null' => TRUE,
        ],
        'age' => [
          'type' => 'int',
          'not null' => FALSE,
        ],
        'price' => [
          'type' => 'numeric',
          'precision' => 10,
          'scale' => 2,
          'not null' => FALSE,
        ],
        'data' => [
          'type' => 'blob',
          'size' => 'big',
          'not null' => FALSE,
        ],
      ],
      'primary key' => ['id'],
    ];

    $schema->createTable('test_column_info', $table_spec);

    try {
      $info = $schema->queryColumnInformation('test_column_info');

      // Verify structure.
      $this->assertArrayHasKey('columns', $info);
      $this->assertArrayHasKey('columns_clean', $info);
      $this->assertArrayHasKey('blobs', $info);

      // Verify all columns are present.
      $this->assertArrayHasKey('id', $info['columns']);
      $this->assertArrayHasKey('name', $info['columns']);
      $this->assertArrayHasKey('age', $info['columns']);
      $this->assertArrayHasKey('price', $info['columns']);
      $this->assertArrayHasKey('data', $info['columns']);

      // Verify blob is identified.
      $this->assertArrayHasKey('data', $info['blobs']);
      $this->assertTrue($info['blobs']['data']);

      // Verify column details have expected properties.
      $id_column = $info['columns']['id'];
      $this->assertArrayHasKey('name', $id_column);
      $this->assertArrayHasKey('type', $id_column);
      $this->assertArrayHasKey('is_identity', $id_column);

      // Identity column should be marked.
      $this->assertEquals(1, $id_column['is_identity']);
    }
    finally {
      $schema->dropTable('test_column_info');
    }
  }

  /**
   * Tests that cache reset works correctly.
   */
  public function testCacheReset() {
    $connection = Database::getConnection();
    /** @var \Drupal\sqlsrv\Driver\Database\sqlsrv\Schema $schema */
    $schema = $connection->schema();

    $connection_options = $connection->getConnectionOptions();
    $cache_enabled = !empty($connection_options['cache_schema']);

    if (!$cache_enabled) {
      $this->markTestSkipped('This test requires cache_schema to be enabled.');
    }

    // Create a test table.
    $table_spec = [
      'fields' => [
        'id' => [
          'type' => 'serial',
          'not null' => TRUE,
        ],
        'name' => [
          'type' => 'varchar',
          'length' => 255,
          'not null' => TRUE,
        ],
      ],
      'primary key' => ['id'],
    ];

    $schema->createTable('test_cache_reset', $table_spec);

    try {
      // Query to populate cache.
      $info1 = $schema->queryColumnInformation('test_cache_reset');
      $this->assertCount(2, $info1['columns']);

      // Reset the cache.
      $schema->resetColumnInformation('test_cache_reset');

      // Query again - should re-query the database.
      $info2 = $schema->queryColumnInformation('test_cache_reset');

      // Should still have the same structure.
      $this->assertEquals($info1, $info2);
      $this->assertCount(2, $info2['columns']);
    }
    finally {
      $schema->dropTable('test_cache_reset');
    }
  }

  /**
   * Tests that cache is invalidated when table is dropped.
   */
  public function testCacheInvalidationOnDrop() {
    $connection = Database::getConnection();
    /** @var \Drupal\sqlsrv\Driver\Database\sqlsrv\Schema $schema */
    $schema = $connection->schema();

    // Create a test table.
    $table_spec = [
      'fields' => [
        'id' => [
          'type' => 'serial',
          'not null' => TRUE,
        ],
        'name' => [
          'type' => 'varchar',
          'length' => 255,
          'not null' => TRUE,
        ],
      ],
      'primary key' => ['id'],
    ];

    $schema->createTable('test_cache_drop', $table_spec);

    // Query to populate cache.
    $info = $schema->queryColumnInformation('test_cache_drop');
    $this->assertArrayHasKey('id', $info['columns']);

    // Drop the table.
    $schema->dropTable('test_cache_drop');

    // Querying a non-existent table should return empty array.
    $info_after_drop = $schema->queryColumnInformation('test_cache_drop');
    $this->assertEmpty($info_after_drop);
  }

}
