<?php

namespace Drupal\Tests\sqlsrv\Kernel;

use Drupal\Core\Database\Database;

/**
 * Tests custom schema creation and handling.
 *
 * This test validates that the sqlsrv driver can properly:
 * - Check if a schema exists
 * - Create a new schema
 * - Create tables in a custom schema
 * - Query tables in a custom schema.
 *
 * @group Database
 */
class SchemaCreationTest extends SqlsrvTestBase {

  /**
   * Tests that schemaExists() correctly identifies existing schemas.
   */
  public function testSchemaExists() {
    $connection = Database::getConnection();
    /** @var \Drupal\sqlsrv\Driver\Database\sqlsrv\Schema $schema */
    $schema = $connection->schema();

    // The default schema (dbo) should always exist.
    $this->assertTrue($schema->schemaExists('dbo'), 'Default dbo schema should exist.');

    // A random schema should not exist.
    $this->assertFalse($schema->schemaExists('nonexistent_schema_12345'), 'Non-existent schema should return FALSE.');
  }

  /**
   * Tests creating a new schema.
   */
  public function testCreateSchema() {
    $connection = Database::getConnection();
    /** @var \Drupal\sqlsrv\Driver\Database\sqlsrv\Schema $schema */
    $schema = $connection->schema();

    $test_schema = 'test_drupal_schema';

    // Clean up if it exists from a previous test run.
    if ($schema->schemaExists($test_schema)) {
      $connection->queryDirect("DROP SCHEMA [{$test_schema}]");
    }

    // Verify it doesn't exist.
    $this->assertFalse($schema->schemaExists($test_schema), 'Test schema should not exist initially.');

    // Create the schema.
    $schema->createSchema($test_schema);

    // Verify it now exists.
    $this->assertTrue($schema->schemaExists($test_schema), 'Test schema should exist after creation.');

    // Clean up.
    $connection->queryDirect("DROP SCHEMA [{$test_schema}]");
  }

  /**
   * Tests that invalid schema names are rejected.
   */
  public function testInvalidSchemaName() {
    $connection = Database::getConnection();
    /** @var \Drupal\sqlsrv\Driver\Database\sqlsrv\Schema $schema */
    $schema = $connection->schema();

    $invalid_names = [
      'test; DROP TABLE users--',
      'test schema',
      'test-schema',
      'test.schema',
      'test[schema]',
    ];

    foreach ($invalid_names as $invalid_name) {
      $caught_exception = FALSE;
      try {
        $schema->createSchema($invalid_name);
      }
      catch (\InvalidArgumentException $e) {
        $caught_exception = TRUE;
        $this->assertStringContainsString('Invalid schema name', $e->getMessage());
      }
      $this->assertTrue($caught_exception, "Invalid schema name '{$invalid_name}' should throw InvalidArgumentException.");
    }
  }

  /**
   * Tests creating tables in a custom schema.
   */
  public function testCreateTableInCustomSchema() {
    $connection = Database::getConnection();
    /** @var \Drupal\sqlsrv\Driver\Database\sqlsrv\Schema $schema */
    $schema = $connection->schema();

    $test_schema = 'test_custom_schema';

    // Clean up if exists from previous test.
    if ($schema->schemaExists($test_schema)) {
      try {
        $connection->queryDirect("DROP TABLE [{$test_schema}].[test_table]");
      }
      catch (\Exception $e) {
        // Table might not exist, ignore.
      }
      $connection->queryDirect("DROP SCHEMA [{$test_schema}]");
    }

    // Create the custom schema.
    $schema->createSchema($test_schema);

    // Create a table directly in the custom schema (bypassing Drupal's
    // schema handling).
    $connection->queryDirect("
      CREATE TABLE [{$test_schema}].[test_table] (
        id INT PRIMARY KEY,
        name VARCHAR(255)
      )
    ");

    // Insert data.
    $connection->queryDirect("
      INSERT INTO [{$test_schema}].[test_table] (id, name)
      VALUES (1, 'Test Record')
    ");

    // Query data.
    $result = $connection->queryDirect("
      SELECT name FROM [{$test_schema}].[test_table] WHERE id = 1
    ")->fetchField();

    $this->assertEquals('Test Record', $result, 'Should be able to query data from custom schema.');

    // Clean up.
    $connection->queryDirect("DROP TABLE [{$test_schema}].[test_table]");
    $connection->queryDirect("DROP SCHEMA [{$test_schema}]");
  }

  /**
   * Tests that getDefaultSchema() returns the configured schema.
   */
  public function testGetDefaultSchema() {
    $connection = Database::getConnection();
    /** @var \Drupal\sqlsrv\Driver\Database\sqlsrv\Schema $schema */
    $schema = $connection->schema();

    // Get the default schema.
    $default_schema = $schema->getDefaultSchema();

    // It should return a non-empty string.
    $this->assertIsString($default_schema);
    $this->assertNotEmpty($default_schema);

    // If no custom schema is configured, it should typically be 'dbo'.
    // However, if a custom schema was configured in connection options,
    // it should return that instead.
    $connection_options = $connection->getConnectionOptions();
    if (isset($connection_options['schema'])) {
      $this->assertEquals($connection_options['schema'], $default_schema,
        'getDefaultSchema() should return the configured schema.');
    }
    else {
      // No custom schema configured, should return database default.
      $this->assertNotEmpty($default_schema, 'Default schema should not be empty.');
    }
  }

}
