<?php

namespace Drupal\Tests\sqlsrv\Kernel;

use Drupal\Core\Database\Database;

/**
 * Tests that MERGE queries correctly exclude primary key columns from UPDATE.
 *
 * This test validates the fix for the deadlock issue where primary key
 * columns were being included in the UPDATE SET clause of MERGE statements,
 * causing unnecessary index updates and lock escalation.
 *
 * @group Database
 */
class UpsertPrimaryKeyTest extends SqlsrvTestBase {

  /**
   * Tests that VARCHAR primary keys are excluded from UPDATE SET clause.
   *
   * Cache tables use VARCHAR primary keys (not identity columns).
   * The bug was that cid was included in UPDATE SET, causing deadlocks.
   */
  public function testVarcharPrimaryKeyExcludedFromUpdate() {
    $connection = Database::getConnection();

    // Create a cache-like table with VARCHAR primary key (no identity).
    $table_spec = [
      'fields' => [
        'cid' => [
          'type' => 'varchar',
          'length' => 255,
          'not null' => TRUE,
        ],
        'data' => [
          'type' => 'blob',
          'size' => 'big',
          'not null' => FALSE,
        ],
        'expire' => [
          'type' => 'int',
          'not null' => TRUE,
          'default' => 0,
        ],
        'created' => [
          'type' => 'numeric',
          'precision' => 14,
          'scale' => 3,
          'not null' => TRUE,
          'default' => 0,
        ],
      ],
      'primary key' => ['cid'],
    ];

    $connection->schema()->createTable('test_cache_upsert', $table_spec);

    try {
      // Create an upsert query.
      $upsert = $connection->upsert('test_cache_upsert')
        ->key('cid')
        ->fields(['cid', 'data', 'expire', 'created']);

      // Add a row.
      $upsert->values([
        'cid' => 'test:key:1',
        'data' => serialize(['test' => 'data']),
        'expire' => -1,
        'created' => 123.456,
      ]);

      // Get the generated SQL.
      $query = (string) $upsert;

      // Extract just the UPDATE SET clause.
      preg_match('/UPDATE SET (.+?) WHEN NOT MATCHED/s', $query, $matches);
      $update_clause = $matches[1] ?? '';

      // Assert that cid is NOT in the UPDATE SET clause.
      $this->assertStringNotContainsString('[cid]', $update_clause, 'Primary key cid should not be in UPDATE SET clause.');

      // Assert that other columns ARE in the UPDATE SET clause.
      $this->assertStringContainsString('[expire]=src.[expire]', str_replace(' ', '', $update_clause), 'Non-primary key columns should be in UPDATE SET clause.');
      $this->assertStringContainsString('[created]=src.[created]', str_replace(' ', '', $update_clause), 'Non-primary key columns should be in UPDATE SET clause.');

      // Execute the upsert (insert).
      $result = $upsert->execute();
      $this->assertIsInt($result);
      $this->assertGreaterThanOrEqual(1, $result);

      // Verify data was inserted.
      $record = $connection->select('test_cache_upsert', 'c')
        ->fields('c')
        ->condition('cid', 'test:key:1')
        ->execute()
        ->fetchObject();

      $this->assertEquals('test:key:1', $record->cid);
      $this->assertEquals(-1, $record->expire);

      // Now update the same record (MERGE should use WHEN MATCHED path).
      $upsert2 = $connection->upsert('test_cache_upsert')
        ->key('cid')
        ->fields(['cid', 'data', 'expire', 'created']);

      $upsert2->values([
        'cid' => 'test:key:1',
        'data' => serialize(['test' => 'updated']),
        'expire' => 999,
        'created' => 456.789,
      ]);

      $result2 = $upsert2->execute();
      $this->assertIsInt($result2);

      // Verify data was updated.
      $record2 = $connection->select('test_cache_upsert', 'c')
        ->fields('c')
        ->condition('cid', 'test:key:1')
        ->execute()
        ->fetchObject();

      $this->assertEquals('test:key:1', $record2->cid, 'Primary key should not change.');
      $this->assertEquals(999, $record2->expire, 'Expire should be updated.');
      $this->assertEquals('456.789', $record2->created, 'Created should be updated.');

    }
    finally {
      $connection->schema()->dropTable('test_cache_upsert');
    }
  }

  /**
   * Tests that identity columns still trigger IDENTITY_INSERT correctly.
   *
   * Node tables use BIGINT identity columns as primary keys.
   * This test ensures the fix doesn't break identity column handling.
   */
  public function testIdentityPrimaryKeyHandling() {
    $connection = Database::getConnection();

    // Create a node-like table with BIGINT identity primary key.
    $table_spec = [
      'fields' => [
        'nid' => [
          'type' => 'serial',
          'not null' => TRUE,
        ],
        'title' => [
          'type' => 'varchar',
          'length' => 255,
          'not null' => TRUE,
        ],
        'status' => [
          'type' => 'int',
          'size' => 'tiny',
          'not null' => TRUE,
          'default' => 1,
        ],
      ],
      'primary key' => ['nid'],
    ];

    $connection->schema()->createTable('test_node_upsert', $table_spec);

    try {
      // Upsert with explicit nid (should enable IDENTITY_INSERT).
      $upsert = $connection->upsert('test_node_upsert')
        ->key('nid')
        ->fields(['nid', 'title', 'status']);

      $upsert->values([
        'nid' => 100,
        'title' => 'Test Node',
        'status' => 1,
      ]);

      // Get the generated SQL.
      $query = (string) $upsert;

      // Extract just the UPDATE SET clause.
      preg_match('/UPDATE SET (.+?) WHEN NOT MATCHED/s', $query, $matches);
      $update_clause = $matches[1] ?? '';

      // Assert that nid (identity + PK) is NOT in UPDATE SET clause.
      $this->assertStringNotContainsString('[nid]', $update_clause, 'Identity primary key should not be in UPDATE SET clause.');

      // Execute should work without IDENTITY_INSERT error.
      $result = $upsert->execute();
      $this->assertIsInt($result);
      $this->assertGreaterThanOrEqual(1, $result);

      // Verify the record was inserted with the specified nid.
      $record = $connection->select('test_node_upsert', 'n')
        ->fields('n')
        ->condition('nid', 100)
        ->execute()
        ->fetchObject();

      $this->assertEquals(100, $record->nid, 'Identity column should have the specified value.');
      $this->assertEquals('Test Node', $record->title);

      // Update the same record.
      $upsert2 = $connection->upsert('test_node_upsert')
        ->key('nid')
        ->fields(['nid', 'title', 'status']);

      $upsert2->values([
        'nid' => 100,
        'title' => 'Updated Node',
        'status' => 0,
      ]);

      $result2 = $upsert2->execute();
      $this->assertIsInt($result2);

      // Verify the record was updated.
      $record2 = $connection->select('test_node_upsert', 'n')
        ->fields('n')
        ->condition('nid', 100)
        ->execute()
        ->fetchObject();

      $this->assertEquals(100, $record2->nid, 'Identity primary key should not change.');
      $this->assertEquals('Updated Node', $record2->title, 'Title should be updated.');
      $this->assertEquals(0, $record2->status, 'Status should be updated.');

    }
    finally {
      $connection->schema()->dropTable('test_node_upsert');
    }
  }

  /**
   * Tests composite primary keys are excluded from UPDATE SET clause.
   */
  public function testCompositePrimaryKeyExcludedFromUpdate() {
    $connection = Database::getConnection();

    // Create a table with composite primary key.
    $table_spec = [
      'fields' => [
        'entity_type' => [
          'type' => 'varchar',
          'length' => 32,
          'not null' => TRUE,
        ],
        'entity_id' => [
          'type' => 'int',
          'not null' => TRUE,
        ],
        'langcode' => [
          'type' => 'varchar',
          'length' => 12,
          'not null' => TRUE,
        ],
        'data' => [
          'type' => 'blob',
          'size' => 'big',
          'not null' => FALSE,
        ],
      ],
      'primary key' => ['entity_type', 'entity_id', 'langcode'],
    ];

    $connection->schema()->createTable('test_composite_upsert', $table_spec);

    try {
      // Note: Upsert with composite keys uses only the first key field.
      $upsert = $connection->upsert('test_composite_upsert')
        ->key('entity_type')
        ->fields(['entity_type', 'entity_id', 'langcode', 'data']);

      $upsert->values([
        'entity_type' => 'node',
        'entity_id' => 1,
        'langcode' => 'en',
        'data' => serialize(['test' => 'data']),
      ]);

      $query = (string) $upsert;

      // Extract just the UPDATE SET clause.
      preg_match('/UPDATE SET (.+?) WHEN NOT MATCHED/s', $query, $matches);
      $update_clause = $matches[1] ?? '';

      // The key field (entity_type) should not be in UPDATE SET.
      $this->assertStringNotContainsString('[entity_type]', $update_clause, 'Key field should not be in UPDATE SET clause.');

      // Execute the upsert.
      $result = $upsert->execute();
      $this->assertIsInt($result);
      $this->assertGreaterThanOrEqual(1, $result);

    }
    finally {
      $connection->schema()->dropTable('test_composite_upsert');
    }
  }

  /**
   * Tests that findPrimaryKeyColumns() method works correctly.
   *
   * This validates that the Schema::findPrimaryKeyColumns() method
   * correctly identifies all primary key columns, not just identities.
   */
  public function testFindPrimaryKeyColumns() {
    $connection = Database::getConnection();
    /** @var \Drupal\sqlsrv\Driver\Database\sqlsrv\Schema $schema */
    $schema = $connection->schema();

    // Test with cache_entity table (VARCHAR PK, no identity).
    if ($schema->tableExists('cache_entity')) {
      $pk_columns = $schema->findPrimaryKeyColumns('cache_entity');
      $this->assertIsArray($pk_columns);
      $this->assertContains('cid', $pk_columns, 'cid should be identified as primary key.');
      $this->assertCount(1, $pk_columns, 'cache_entity should have one PK column.');
    }

    // Test with node table (BIGINT identity PK).
    if ($schema->tableExists('node')) {
      $pk_columns = $schema->findPrimaryKeyColumns('node');
      $this->assertIsArray($pk_columns);
      $this->assertContains('nid', $pk_columns, 'nid should be identified as primary key.');
    }

    // Test with test_people table (VARCHAR PK 'job').
    if ($schema->tableExists('test_people')) {
      $pk_columns = $schema->findPrimaryKeyColumns('test_people');
      $this->assertIsArray($pk_columns);
      $this->assertContains('job', $pk_columns, 'job should be identified as primary key.');
    }
  }

}
