<?php

namespace Drupal\Tests\sqlsrv\Unit;

use Drupal\sqlsrv\Driver\Database\sqlsrv\Condition;
use Drupal\Tests\UnitTestCase;

/**
 * Tests that the driver-specific Condition class exists in correct namespace.
 *
 * This test replaces four previous kernel tests (SelectTest, MergeTest,
 * DeleteTruncateTest, UpdateTest) that each tested namespace correctness
 * using reflection on query builders with database connections. Those tests
 * were removed during D9 compatibility cleanup.
 *
 * Why this test is important?
 *
 * Query builders (Select, Delete, Update, Merge) use this Condition class
 * to provide SQL Server-specific behavior:
 *
 * 1. LIKE operator escaping: SQL Server uses brackets [%] instead of
 *    backslashes \% due to a PDO bug (https://bugs.php.net/bug.php?id=79276)
 *    See SqlsrvConditionTest for detailed LIKE escaping tests.
 *
 * 2. REGEXP operator translation: Converts MySQL REGEXP syntax to SQL Server
 *    REGEXP() function calls with proper parameter ordering.
 *    Example: "name REGEXP '^P'" becomes "(REGEXP('^P', name) = 1)"
 *
 * If the driver didn't provide its own Condition class, query builders would
 * fall back to core's Condition class, causing LIKE and REGEXP queries to
 * fail on SQL Server.
 *
 * This test verifies the architecture is correct without requiring database
 * connections or query execution.
 *
 * @see \Drupal\Tests\sqlsrv\Unit\SqlsrvConditionTest
 * @see \Drupal\sqlsrv\Driver\Database\sqlsrv\Condition::condition()
 *
 * @group Database
 */
class ConditionNamespaceTest extends UnitTestCase {

  /**
   * Tests that the Condition class exists in the driver namespace.
   */
  public function testConditionClassExists() {
    $this->assertTrue(class_exists(Condition::class),
      'Driver-specific Condition class should exist.');
  }

  /**
   * Tests that the Condition class is in the correct namespace.
   */
  public function testConditionNamespace() {
    $expectedNamespace = 'Drupal\sqlsrv\Driver\Database\sqlsrv\Condition';
    $this->assertSame($expectedNamespace, Condition::class,
      'Condition class should be in the driver namespace.');
  }

  /**
   * Tests that the Condition class extends core Condition.
   */
  public function testConditionExtendsCore() {
    $reflection = new \ReflectionClass(Condition::class);
    $parentClass = $reflection->getParentClass();

    $this->assertNotFalse($parentClass,
      'Condition should extend a parent class.');

    $this->assertSame('Drupal\Core\Database\Query\Condition', $parentClass->getName(),
      'Condition should extend core Drupal Condition class.');
  }

  /**
   * Tests that Condition can be instantiated.
   */
  public function testConditionCanBeInstantiated() {
    $condition = new Condition('AND');
    $this->assertInstanceOf(Condition::class, $condition,
      'Should be able to instantiate Condition class.');

    $this->assertInstanceOf('Drupal\Core\Database\Query\Condition', $condition,
      'Condition should be an instance of core Condition.');
  }

}
