<?php

namespace Drupal\sqlsrv\Driver\Database\sqlsrv;

use Drupal\Core\Database\Connection as DatabaseConnection;
use Drupal\Core\Database\Query\Condition as QueryCondition;
use Drupal\Core\Database\Query\PlaceholderInterface;

/**
 * @addtogroup database
 * @{
 */
class Condition extends QueryCondition {

  /**
   * {@inheritdoc}
   *
   * Overridden to replace REGEXP expressions and handle LIKE BINARY.
   */
  public function compile(DatabaseConnection $connection, PlaceholderInterface $queryPlaceholder) {
    /** @var \Drupal\sqlsrv\Driver\Database\sqlsrv\Connection $connection */
    // Find any REGEXP conditions and turn them into function calls.
    foreach ($this->conditions as &$condition) {
      if (isset($condition['operator'])) {
        if ($condition['operator'] == 'REGEXP' || $condition['operator'] == 'NOT REGEXP') {
          $placeholder = ':db_condition_placeholder_' . $queryPlaceholder->nextPlaceholder();
          $field_fragment = $connection->escapeField($condition['field']);
          $comparison = $condition['operator'] == 'REGEXP' ? '1' : '0';
          $condition['field'] = "REGEXP({$placeholder}, {$field_fragment}) = {$comparison}";
          $condition['operator'] = NULL;
          $condition['value'] = [$placeholder => $condition['value']];
        }
        // Handle LIKE BINARY for case-sensitive matching on CI databases.
        elseif ($condition['operator'] == 'LIKE BINARY') {
          // For case-insensitive databases, we need to force case-sensitive
          // comparison by adding a COLLATE clause to the field.
          $schema = $connection->schema();
          if (!$schema->isCaseSensitive()) {
            $collation = $schema->getCollation();
            // Replace _CI_ with _CS_ to get case-sensitive collation.
            $cs_collation = preg_replace("/_C[IS]_/", "_CS_", $collation);
            // Escape the field and create COLLATE expression.
            $field_name = $condition['field'];
            $field_escaped = $connection->escapeField($field_name);
            // Create placeholder for value and escape it for LIKE.
            $value = $condition['value'];
            $value = str_replace('[', '[[]', $value);
            $value = str_replace('\%', '[%]', $value);
            $value = str_replace('\_', '[_]', $value);
            $value = str_replace('\\\\', '\\', $value);
            // Build the complete expression as a WHERE snippet.
            $value_placeholder = ':db_condition_placeholder_' . $queryPlaceholder->nextPlaceholder();
            $condition['field'] = "{$field_escaped} COLLATE {$cs_collation} LIKE {$value_placeholder}";
            $condition['value'] = [$value_placeholder => $value];
            $condition['operator'] = NULL;
            continue;
          }
          // Change operator to regular LIKE for CS databases.
          $condition['operator'] = 'LIKE';
        }
        // Drupal expects all LIKE expressions to be escaped with a backslash.
        // Due to a PDO bug sqlsrv uses its default escaping behavior.
        // This can be removed if https://bugs.php.net/bug.php?id=79276 is
        // fixed.
        if ($condition['operator'] == 'LIKE' || $condition['operator'] == 'NOT LIKE') {
          // Use sequential replacements to handle escapes correctly.
          // Order matters: [ must be escaped first, then \% and \_, then \\.
          $value = $condition['value'];
          $value = str_replace('[', '[[]', $value);
          $value = str_replace('\%', '[%]', $value);
          $value = str_replace('\_', '[_]', $value);
          $value = str_replace('\\\\', '\\', $value);
          $condition['value'] = $value;
        }
      }
    }
    parent::compile($connection, $queryPlaceholder);
  }

  /**
   * {@inheritdoc}
   *
   * Overridden to replace REGEXP expressions.
   * Needs to be tested for complex nested expressions.
   */
  public function where($snippet, $args = []) {
    $operator = NULL;
    if (strpos($snippet, " NOT REGEXP ") !== FALSE) {
      $operator = ' NOT REGEXP ';
    }
    elseif (strpos($snippet, " REGEXP ") !== FALSE) {
      $operator = ' REGEXP ';
    }
    if ($operator !== NULL) {
      $fragments = explode($operator, $snippet);
      $field = $fragments[0];
      $value = $fragments[1];
      $comparison = $operator == ' REGEXP ' ? '1' : '0';

      $snippet = "REGEXP({$value}, {$field}) = {$comparison}";
      $operator = NULL;
    }
    $this->conditions[] = [
      'field' => $snippet,
      'value' => $args,
      'operator' => $operator,
    ];
    $this->changed = TRUE;
    return $this;
  }

}

/**
 * @} End of "addtogroup database".
 */
