<?php

namespace Drupal\Tests\sqlsrv\Kernel;

use Drupal\Core\Database\Database;
use Drupal\KernelTests\Core\Database\DatabaseTestBase;

/**
 * Test aliases within GROUP BY and ORDER BY.
 *
 * @group Database
 * @requires extension pdo_sqlsrv
 */
abstract class SqlsrvTestBase extends DatabaseTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = ['database_test', 'sqlsrv'];

  /**
   * {@inheritdoc}
   *
   * Skip any kernel tests if not running on the correct database.
   */
  protected function setUp():void {
    // Skip tests if pdo_sqlsrv extension is not available.
    if (!extension_loaded('pdo_sqlsrv')) {
      // Set siteDirectory to prevent tearDown errors.
      $this->siteDirectory = 'vfs://';
      $this->markTestSkipped('The pdo_sqlsrv PHP extension is not available.');
      return;
    }
    parent::setUp();
    // Skip if not connected to SQL Server.
    // This allows the test to work with any database configuration
    // and only run when SQL Server is actually configured.
    if (Database::getConnection()->databaseType() !== 'sqlsrv') {
      $this->markTestSkipped("This test only runs for MS SQL Server. Configure SIMPLETEST_DB in phpunit.xml or your environment.");
    }
  }

  /**
   * Get SQL Server test credentials from environment.
   *
   * Parses SIMPLETEST_DB or uses explicit environment variables.
   *
   * @return array
   *   An array with keys: host, username, password, and database.
   */
  protected function getSqlServerCredentials(): array {
    // Try explicit environment variables first.
    $host = getenv('SQLSRV_TEST_HOST');
    $username = getenv('SQLSRV_TEST_USERNAME');
    $password = getenv('SQLSRV_TEST_PASSWORD');

    // If not set, parse from SIMPLETEST_DB.
    if (!$host || !$username || !$password) {
      $simpletest_db = getenv('SIMPLETEST_DB');
      if ($simpletest_db) {
        $parsed = parse_url($simpletest_db);
        $host = $host ?: ($parsed['host'] ?? 'sqlsrv');
        $username = $username ?: ($parsed['user'] ?? 'sa');
        $password = $password ?: ($parsed['pass'] ?? '');
        $database = $parsed['path'] ? ltrim($parsed['path'], '/') : 'master';
      }
      else {
        // Fallback defaults for tests.
        $host = $host ?: 'sqlsrv';
        $username = $username ?: 'sa';
        $password = $password ?: '';
        $database = 'master';
      }
    }
    else {
      $database = getenv('SQLSRV_TEST_DATABASE') ?: 'master';
    }

    return [
      'host' => $host,
      'username' => $username,
      'password' => $password,
      'database' => $database,
    ];
  }

}
