<?php

namespace Drupal\Tests\sqlsrv\Kernel;

use Drupal\Core\Database\Database;
use Drupal\sqlsrv\Driver\Database\sqlsrv\Utils;

/**
 * Tests UTF-8 encoding handling for SQL Server driver.
 *
 * SQL Server's PDO driver requires all strings to be valid UTF-8 as they
 * are converted to UTF-16 internally. This test validates that invalid
 * UTF-8 sequences are properly handled.
 *
 * @group Database
 */
class Utf8EncodingTest extends SqlsrvTestBase {

  /**
   * Tests that ensureValidUtf8() correctly validates and cleans strings.
   */
  public function testEnsureValidUtf8() {
    // Test valid UTF-8 passes through unchanged.
    $valid = 'Hello World!';
    $this->assertEquals($valid, Utils::ensureValidUtf8($valid));

    // Test valid UTF-8 with multibyte characters.
    $multibyte = 'Hello 世界!';
    $this->assertEquals($multibyte, Utils::ensureValidUtf8($multibyte));

    // Test that non-strings pass through unchanged.
    $this->assertEquals(123, Utils::ensureValidUtf8(123));
    $this->assertEquals(NULL, Utils::ensureValidUtf8(NULL));
    $this->assertEquals(TRUE, Utils::ensureValidUtf8(TRUE));

    // Test invalid UTF-8 byte sequence is cleaned.
    // Create an invalid UTF-8 string (0xC0 0x80 is invalid).
    $invalid = "Hello" . chr(0xC0) . chr(0x80) . "World";
    $cleaned = Utils::ensureValidUtf8($invalid);
    // The result should be valid UTF-8.
    $this->assertTrue(mb_check_encoding($cleaned, 'UTF-8'));
  }

  /**
   * Tests database operations with valid UTF-8 strings.
   */
  public function testValidUtf8Insert() {
    $connection = Database::getConnection();
    /** @var \Drupal\sqlsrv\Driver\Database\sqlsrv\Schema $schema */
    $schema = $connection->schema();

    // Create a test table.
    $table_spec = [
      'fields' => [
        'id' => [
          'type' => 'serial',
          'not null' => TRUE,
        ],
        'name' => [
          'type' => 'varchar',
          'length' => 255,
          'not null' => TRUE,
        ],
      ],
      'primary key' => ['id'],
    ];

    $schema->createTable('test_utf8_encoding', $table_spec);

    try {
      // Test valid UTF-8 multibyte characters.
      $multibyte_strings = [
        'Hello 世界',
        'Привет мир',
        'مرحبا بالعالم',
        'Emoji test 😀🎉',
      ];

      foreach ($multibyte_strings as $string) {
        $id = $connection->insert('test_utf8_encoding')
          ->fields(['name' => $string])
          ->execute();

        $result = $connection->select('test_utf8_encoding', 't')
          ->fields('t', ['name'])
          ->condition('id', $id)
          ->execute()
          ->fetchField();

        $this->assertEquals($string, $result, "Failed to store and retrieve: $string");
      }
    }
    finally {
      $schema->dropTable('test_utf8_encoding');
    }
  }

  /**
   * Tests that URL-encoded strings don't cause UTF-16 errors.
   */
  public function testUrlEncodedStrings() {
    $connection = Database::getConnection();
    /** @var \Drupal\sqlsrv\Driver\Database\sqlsrv\Schema $schema */
    $schema = $connection->schema();

    // Create a test table.
    $table_spec = [
      'fields' => [
        'id' => [
          'type' => 'serial',
          'not null' => TRUE,
        ],
        'path' => [
          'type' => 'varchar',
          'length' => 255,
          'not null' => TRUE,
        ],
      ],
      'primary key' => ['id'],
    ];

    $schema->createTable('test_url_encoding', $table_spec);

    try {
      // Test URL-like strings that might contain percent-encoded characters.
      // The %cd pattern from the issue would be handled here.
      $url_strings = [
        '/path/to/page',
        '/search?q=test',
        '/user/login',
        '/%cd',
        // This is actually valid UTF-8, but let's test it.
      ];

      foreach ($url_strings as $path) {
        $id = $connection->insert('test_url_encoding')
          ->fields(['path' => $path])
          ->execute();

        $result = $connection->select('test_url_encoding', 't')
          ->fields('t', ['path'])
          ->condition('id', $id)
          ->execute()
          ->fetchField();

        $this->assertEquals($path, $result, "Failed to store and retrieve URL: $path");
      }

      // Test querying with URL pattern.
      $result = $connection->select('test_url_encoding', 't')
        ->fields('t', ['path'])
        ->condition('path', '/%', 'LIKE')
        ->execute()
        ->fetchAll();

      // Should find the /%cd path.
      $this->assertGreaterThan(0, count($result), 'Should find paths starting with /');
    }
    finally {
      $schema->dropTable('test_url_encoding');
    }
  }

  /**
   * Tests UPDATE queries with potentially invalid UTF-8.
   */
  public function testUpdateWithUtf8() {
    $connection = Database::getConnection();
    /** @var \Drupal\sqlsrv\Driver\Database\sqlsrv\Schema $schema */
    $schema = $connection->schema();

    // Create a test table.
    $table_spec = [
      'fields' => [
        'id' => [
          'type' => 'serial',
          'not null' => TRUE,
        ],
        'name' => [
          'type' => 'varchar',
          'length' => 255,
          'not null' => TRUE,
        ],
      ],
      'primary key' => ['id'],
    ];

    $schema->createTable('test_update_utf8', $table_spec);

    try {
      // Insert initial value.
      $id = $connection->insert('test_update_utf8')
        ->fields(['name' => 'Initial Value'])
        ->execute();

      // Update with multibyte string.
      $new_value = 'Updated 世界';
      $connection->update('test_update_utf8')
        ->fields(['name' => $new_value])
        ->condition('id', $id)
        ->execute();

      // Verify update.
      $result = $connection->select('test_update_utf8', 't')
        ->fields('t', ['name'])
        ->condition('id', $id)
        ->execute()
        ->fetchField();

      $this->assertEquals($new_value, $result);
    }
    finally {
      $schema->dropTable('test_update_utf8');
    }
  }

}
