<?php

namespace Drupal\sri_ui\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;

/**
 * Configure AddToAny settings for this site.
 */
class AdminSettingsForm extends ConfigFormBase {

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'sri_ui_settings_form';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return [
      'sri_ui.settings',
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $sri_ui_settings = $this->config('sri_ui.settings');
    $assets = $sri_ui_settings->get('assets') ?? [];
    $status = $sri_ui_settings->get('status');
    $refresh_timout = $sri_ui_settings->get('key_refresh_timout') ?? '';
    $last_refresh_timout = $sri_ui_settings->get('last_refresh_timout') ?? time();
    $form['status'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable custom Sri integrity attributes for 3rd party assets (js and css).'),
      '#default_value' => $status ?? FALSE,
    ];
    $form['key_refresh_timout'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Timeout for next refresh.'),
      '#description' => $this->t('Enter intravel time in seconds (Ex. 1 Minute to 60, 1 Hour to 3600, 1 Day to 604800).'),
      '#default_value' => $refresh_timout,
      '#element_validate' => ['::validateTimestamp'],
    ];
    $form['last_refresh_timout'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Last refresh timeout'),
      '#default_value' => $last_refresh_timout,
      '#access' => FALSE,
    ];
    $form['assets'] = ['#tree' => TRUE];
    foreach ($assets as $asset) {
      $form['assets'][] = [
        '#type' => 'details',
        '#title' => $this->t('SRI integrity config for %asset', ['%asset' => $asset['asset']]),
        '#open' => FALSE,
      ] + $this->getAssetsFieds($asset['asset'], $asset);
    }
    $form['assets'][] = [
      '#type' => 'details',
      '#title' => $this->t('New Asset'),
      '#open' => TRUE,
    ] + $this->getAssetsFieds('New', []);
    $form['markup'] = [
      '#type' => 'markup',
      '#markup' => 'Note: Please clear all caches to reflect new sri option in frontend after saving this form.
       New asset form option after saving this form.',
    ];
    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $config = $this->config('sri_ui.settings');
    $values = $form_state->getValues();
    $assets = [];
    $i = 0;
    foreach ($values['assets'] as $asset) {
      $asset = array_map('trim', $asset);
      if ($asset['asset'] !== '') {
        $assets[$i]['is_check_update'] = $asset['is_check_update'];
        $assets[$i]['add_async'] = $asset['add_async'];
        $assets[$i]['asset'] = $asset['asset'];
        if ($asset['integrity'] !== '') {
          $assets[$i]['integrity'] = $asset['integrity'];
        }
        if ($asset['crossorigin'] !== '') {
          $assets[$i]['crossorigin'] = $asset['crossorigin'];
        }
        if ($asset['query_string'] !== '') {
          $assets[$i]['query_string'] = $asset['query_string'];
        }
        $i++;
      }
    }
    $config->set('status', $values['status']);
    $config->set('key_refresh_timout', $values['key_refresh_timout']);
    $config->set('last_refresh_timout', $values['last_refresh_timout']);
    $config->set('assets', $assets);
    $config->save();
    parent::submitForm($form, $form_state);
  }

  /**
   * Get all fields for an assets.
   *
   * @param string $key
   *   Asset key.
   * @param array $asset
   *   Asset array.
   *
   * @return array
   *   return 3 textfield array.
   */
  private function getAssetsFieds(string $key, array $asset) {
    $field = [];
    $status = $asset['is_check_update'];
    $field['is_check_update'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Auto hash refresh on page request.'),
      '#default_value' => $status ?? FALSE,
    ];
    $field['add_async'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Add async attribute to the URL.'),
      '#default_value' => $asset['add_async'] ?? FALSE,
    ];
    $field['asset'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Asset full URL'),
      '#default_value' => $key != 'New' ? $key : '',
      '#description' => $this->t('Please add full URL of extrnal asset which added by drupal library e.g.- https://static.addtoany.com/menu/page.js'),
    ];
    $field['integrity'] = [
      '#type' => 'textfield',
      '#title' => $this->t("Asset integrity's valid checksum"),
      '#default_value' => $key != 'New' ? $asset['integrity'] : '',
      '#description' => $this->t("Please add Asset integrity's valid checksum, you can generate it from https://www.srihash.org/"),
    ];
    $field['crossorigin'] = [
      '#type' => 'textfield',
      '#title' => $this->t("Asset integrity's crossorigin"),
      '#default_value' => $key != 'New' ? $asset['crossorigin'] : '',
      '#description' => $this->t("Please add integrity's crossorigin, it can be 'anonymous'"),
    ];
    $field['query_string'] = [
      '#type' => 'textfield',
      '#title' => $this->t("Asset Query String"),
      '#default_value' => $key != 'New' ? $asset['query_string'] : '',
      '#description' => $this->t("Please add query_string in assets to avoid cache, it can be v1"),
    ];

    return $field;
  }

  /**
   * This function will validate the timestamp field.
   */
  public function validateTimestamp(array $element, FormStateInterface $form_state, array &$complete_form) {
    $value = $form_state->getValue($element['#name']);

    if (!is_numeric($value) || floor($value) != $value) {
      $form_state->setError($element, $this->t('Please enter a valid integer for the timestamp field.'));
    }
  }

}
