<?php

namespace Drupal\sri_ui\Services;

use Drupal\Core\Cache\CacheTagsInvalidatorInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;

/**
 * Class HashGeneration.
 *
 * Class to generate the SRI hashes.
 *
 * @package Drupal\sri_ui
 */
class HashGeneration {

  /**
   * The configuration object factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * The editable configuration object for sri_ui settings.
   *
   * @var \Drupal\Core\Config\Config
   */
  protected $assetsConfig;

  /**
   * The logger channel Factory.
   *
   * @var \Drupal\Core\Logger\LoggerChannelFactoryInterface
   */
  protected LoggerChannelFactoryInterface $loggerChannelFactory;

  /**
   * The cache tags invalidator.
   *
   * @var \Drupal\Core\Cache\CacheTagsInvalidatorInterface
   */
  protected $cacheTagsInvalidator;

  /**
   * Constructs a new ImportCommands object.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $configfactory
   *   The configuration object factory.
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $loggerChannelFactory
   *   Logger.
   * @param \Drupal\Core\Cache\CacheTagsInvalidatorInterface $cache_tags_invalidator
   *   Cache tag invalidator.
   */
  public function __construct(
    ConfigFactoryInterface $configfactory,
    LoggerChannelFactoryInterface $loggerChannelFactory,
    CacheTagsInvalidatorInterface $cache_tags_invalidator,
  ) {
    $this->configFactory = $configfactory;
    $this->assetsConfig = $this->configFactory->getEditable('sri_ui.settings');
    $this->loggerChannelFactory = $loggerChannelFactory;
    $this->cacheTagsInvalidator = $cache_tags_invalidator;
  }

  /**
   * Function to generate the sri has key values.
   */
  public function hashKeyGeneration($check_value = '') {

    $assets = $this->assetsConfig->get('assets') ?? [];

    if (isset($assets)) {
      $is_having_changes = 0;
      foreach ($assets as $key => $assects_values) {
        if ($check_value == 'check_one_by_one') {
          $last_refreshed_timout = $this->assetsConfig->get('last_refresh_timout') ?? time();

          $check_key_refresh_time = $last_refreshed_timout + $this->assetsConfig->get('key_refresh_timout');
          if (
            ($check_key_refresh_time <= time()) &&
            (isset($assects_values['is_check_update']) &&
              $assects_values['is_check_update'] == TRUE)
            ) {
            $this->checkSriHashKeys($assets, $key, $assects_values, $is_having_changes);
          }
        }
        else {
          $this->checkSriHashKeys($assets, $key, $assects_values, $is_having_changes);
        }

      }
      // If have any new updates save the assets settings.
      if ($is_having_changes == 1) {
        $this->assetsConfig->set('assets', $assets);
        $this->assetsConfig->set('last_refresh_timout', time());
        $this->assetsConfig->save();
        // Clear the cache.
        if ($check_value == 'check_one_by_one') {
          $this->clearLibraryCache();
        }
      }

    }

  }

  /**
   * Implements hook_sri_hash().
   *
   * Funtion to generate hash key.
   */
  private function keyGenSriHash($assects_links) {

    // Read the external asset url page content.
    $response = file_get_contents("$assects_links");

    // Create a shase6 has valyes.
    $hash = hash('sha256', $response, TRUE);
    $hash_base64 = base64_encode($hash);
    $this->loggerChannelFactory->get('sri_ui')->notice('Updated Has values %title.',
          [
            '%title' => $assects_links . ' On sha256-' . $hash_base64,
          ]);
    return "sha256-$hash_base64";

  }

  /**
   * Implements hook_clear_library_cache().
   *
   * Funtion to Clear the cache.
   */
  private function clearLibraryCache() {
    // Clear the cache for the specified tags.
    $this->cacheTagsInvalidator->invalidateTags(['library_attachments_cache_tag']);
  }

  /**
   * Funtion to check the sri hash keys.
   */
  private function checkSriHashKeys(&$assets, $key, $assects_values, &$is_having_changes) {
    if (isset($assects_values['asset']) && $assects_values['asset'] !== '') {
      $assets[$key]['asset'] = $assects_links = $assects_values['asset'];
    }
    if (isset($assects_values['integrity']) && $assects_values['integrity'] !== '') {
      $assets[$key]['integrity'] = $assects_integrity = $assects_values['integrity'];
    }
    if (isset($assects_values['crossorigin']) && $assects_values['crossorigin'] !== '') {
      $assets[$key]['crossorigin'] = $assects_values['crossorigin'];
    }
    if (isset($assects_values['query_string']) && $assects_values['query_string'] !== '') {
      $assets[$key]['query_string'] = $assects_values['query_string'];
      if (strpos($assects_links, '?')) {
        $assects_links .= '&' . $assects_values['query_string'];
      }
      else {
        $assects_links .= '?' . $assects_values['query_string'];
      }
    }

    $new_assects_integrity = $this->keyGenSriHash($assects_links);
    // Compare old and new hash values are not equal.
    if ($assects_integrity != $new_assects_integrity) {
      $assets[$key]['integrity'] = $new_assects_integrity;
      $is_having_changes = 1;
    }
  }

}
