<?php

namespace Drupal\ssid\Plugin\Field\FieldFormatter;

use Drupal\Core\Field\FieldItemListInterface;
use Drupal\Core\Field\FormatterBase;
use Drupal\Core\Form\FormStateInterface;

/**
 * @FieldFormatter(
 *   id = "scope_serial_default_formatter",
 *   label = @Translation("Default formmater"),
 *   field_types = {
 *     "scope_serial"
 *   }
 * )
 */
class ScopeSerialDefaultFormatter extends FormatterBase {

  /**
   * {@inheritdoc}
   */
  public static function defaultSettings(): array {
    return [
      'properties' => [
        'plugin' => FALSE,
        'scope' => FALSE,
        'serial' => TRUE,
      ],
      'context' => 'label',
    ] + parent::defaultSettings();
  }

  /**
   * {@inheritdoc}
   */
  public function settingsForm(array $form, FormStateInterface $form_state): array {
    $elements['properties'] = [
      '#type' => 'checkboxes',
      '#title' => $this->t('Display properties'),
      '#options' => [
        'plugin' => $this->t('Plugin'),
        'scope' => $this->t('Scope'),
        'serial' => $this->t('Serial'),
      ],
      '#default_value' => array_filter($this->getSetting('properties')),
      '#description' => $this->t('Select which parts of the value to display.'),
    ];
    $elements['context'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Context'),
      '#default_value' => $this->getSetting('context'),
      '#description' => $this->t('Optional strings to identify the usage context (e.g., @context_examples etc.). Use or remove the specific word "@label_keyword" to print the property labels. This will work only if you are using the default theme implementation. You may need to adjust if you override the module template.', [
        '@context_examples' => '"entity display", "views", "my_context"',
        '@label_keyword' => 'label',
      ]),
    ];

    return $elements;
  }

  /**
   * {@inheritdoc}
   */
  public function settingsSummary(): array {
    $settings = $this->getFormattedSettings();

    return [
      $this->t('Display properties: @properties', ['@properties' => !empty($settings['properties']) ? implode(', ', array_values($settings['properties'])) : $this->t('None')]),
      $this->t('Context: @context', ['@context' => $settings['context']]),
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function viewElements(FieldItemListInterface $items, $langcode) {
    $settings = $this->getFormattedSettings();
    $elements = [];
    foreach ($items as $delta => $item) {
      $properties = [];
      foreach ($settings['properties'] as $key => $label) {
        $properties[$key] = [
          'name' => $key,
          'label' => $label,
          'value' => $item->{$key},
        ];
      }
      $elements[$delta] = [
        '#theme' => 'scope_serial_field_value',
        '#properties' => $properties,
        '#context' => $settings['context'],
        '#cache' => ['tags' => $items->getEntity()->getCacheTags()],
      ];
    }

    return $elements;
  }

  /**
   * Get formatted settings.
   */
  public function getFormattedSettings() {
    $settings = [
      'properties' => [],
      'context' => trim($this->getSetting('context')),
    ];
    $available_properties = [
      'plugin' => $this->t('Plugin'),
      'scope' => $this->t('Scope'),
      'serial' => $this->t('Serial'),
    ];
    $selected_properties = array_keys(array_filter($this->getSetting('properties')));
    if (count($selected_properties) > 0) {
      foreach ($selected_properties as $property) {
        if (isset($available_properties[$property])) {
          $settings['properties'][$property] = $available_properties[$property];
        }
      }
    }
    else {
      $settings['properties'] = $available_properties;
    }

    return $settings;
  }

}
