<?php

namespace Drupal\ssid\Plugin\Field\FieldType;

use Drupal\Core\Field\FieldDefinitionInterface;
use Drupal\Core\Field\FieldItemBase;
use Drupal\Core\Field\FieldStorageDefinitionInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\TypedData\DataDefinition;
use Drupal\Core\StringTranslation\StringTranslationTrait;

/**
 * Plugin implementation of the 'scope_serial' field type.
 *
 * @FieldType(
 *   id = "scope_serial",
 *   label = @Translation("Scope Serial"),
 *   description = @Translation("Stores a per-scope serial number."),
 *   category = "ssid",
 *   default_widget = "scope_serial_default_widget",
 *   default_formatter = "scope_serial_default_formatter",
 * )
 */
class ScopeSerial extends FieldItemBase {

  use StringTranslationTrait;

  /**
   * {@inheritdoc}
   */
  public static function schema(FieldStorageDefinitionInterface $field_definition) {
    return [
      'columns' => [
        'plugin' => [
          'type' => 'varchar',
          'length' => 100,
          'not null' => FALSE,
        ],
        'scope' => [
          'type' => 'varchar',
          'length' => (int) $field_definition->getSetting('scope_max_length'),
          'not null' => FALSE,
        ],
        'serial' => [
          'type' => 'int',
          'unsigned' => TRUE,
          'not null' => FALSE,
        ],
      ],
      'indexes' => [
        'scope_serial' => ['scope', 'serial'],
      ],
    ];
  }

  /**
   * {@inheritdoc}
   */
  public static function propertyDefinitions(FieldStorageDefinitionInterface $field_definition) {
    $properties['plugin'] = DataDefinition::create('string')
      ->setLabel(t('Plugin'))
      ->setRequired(TRUE);
    $properties['scope'] = DataDefinition::create('string')
      ->setLabel(t('Scope'))
      ->setRequired(TRUE);
    $properties['serial'] = DataDefinition::create('integer')
      ->setLabel(t('Serial'))
      ->setRequired(TRUE);

    return $properties;
  }

  /**
   * {@inheritdoc}
   */
  public static function defaultStorageSettings() {
    return [
      'scope_max_length' => 30,
      'translatable' => FALSE,
    ] + parent::defaultStorageSettings();
  }

  /**
   * {@inheritdoc}
   */
  public static function generateSampleValue(FieldDefinitionInterface $field_definition) {
    $value = [
      'plugin' => 'daily',
      'scope' => date('Y-m-d'),
      'serial' => 1,
    ];

    return $value;
  }

  /**
   * {@inheritdoc}
   */
  public function storageSettingsForm(array &$form, FormStateInterface $form_state, $has_data) {
    return [
      'scope_max_length' => [
        '#type' => 'number',
        '#title' => $this->t('Maximum scope length'),
        '#default_value' => $this->getSetting('scope_max_length'),
        '#required' => TRUE,
        '#description' => $this->t('The maximum length of the Scope. This does not affect the numeric value.'),
        '#min' => 1,
        '#disabled' => $has_data,
      ],
    ];
  }

  /**
   * {@inheritdoc}
   */
  public static function defaultFieldSettings() {
    return [
      'bundle' => TRUE,
      'scope_plugin' => 'daily',
    ] + parent::defaultFieldSettings();
  }

  /**
   * {@inheritdoc}
   */
  public function fieldSettingsForm(array $form, FormStateInterface $form_state) {
    $scope_plugin_options = self::serialHandler()->getScopeDefinitionNames();
    return [
     'bundle' => [
       '#type' => 'checkbox',
       '#title' => $this->t('Bundle restriction'),
       '#default_value' => $this->getSetting('bundle'),
       '#description' => $this->t('If you are re-using this field on several bundles, keep separated values.'),
     ],
      'scope_plugin' => [
        '#type' => 'select',
        '#title' => $this->t('Scope plugin'),
        '#options' => $scope_plugin_options,
        '#default_value' => $this->getSetting('scope_plugin'),
        '#description' => $this->t('Select the scope logic that determines how serials reset.'),
        '#required' => TRUE,
      ],
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function getValue() {
    $values = [];
    foreach ($this->getFieldDefinition()->getFieldStorageDefinition()->getColumns() as $column_name => $_) {
      $values[$column_name] = $this->get($column_name)->getValue();
    }

    return $values;
  }

  /**
   * {@inheritdoc}
   */
  public function preSave() {
    $entity = $this->getEntity();
    $field_name = $this->getFieldDefinition()->getName();
    $values = $this->getValue();
    // Well. This is to prevent setting this field value externally.
    $original_values = $entity->original ? $entity->original->get($field_name)->getValue() : NULL;
    if (is_array($original_values)) {
      $original_values = reset($original_values);
      if (is_array($original_values) && $this->isValidStorageData($original_values)) {
        $this->setValue($original_values);

        return;
      }
    }
    $generate = is_array($values) ? !$this->isValidStorageData($values) : TRUE;
    if ($generate) {
      try {
        $data = self::serialHandler()->generateSerial($entity, $field_name);
        $this->setValue($data);
      }
      catch (\Throwable $e) {
        self::logger()->error('Error generating serial: @message', ['@message' => $e->getMessage()]);
      }
    }
  }

  /**
   * {@inheritdoc}
   */
  public static function fieldSettingsSummary(FieldDefinitionInterface $field_definition) : array {
    return [
      t('Scope plugin: @plugin (Max. length: @scope_max_length).', [
        '@plugin' => $field_definition->getSetting('scope_plugin'),
        '@scope_max_length' => $field_definition->getSetting('scope_max_length'),
      ]),
      t('Bundle restriction: @bundle.', [
        '@bundle' => $field_definition->getSetting('bundle') ? t('Enabled') : t('Disabled'),
      ]),
    ];
  }

  /**
   * Check if array is a valid value.
   *
   * @param array $data
   *   The data to be validated.
   */
  protected function isValidStorageData(array $data) : bool {
    if (
      !array_diff(['plugin', 'scope', 'serial'], array_keys($data)) &&
      !empty($data['plugin']) &&
      !empty($data['scope']) &&
      !empty($data['serial']) &&
      is_numeric($data['serial'])
    ) {
      return TRUE;
    }

    return FALSE;
  }

  /**
   * Returns the serial handler service.
   */
  public static function serialHandler() {
    return \Drupal::service('ssid.serial_handler');
  }

  /**
   * Return the logger channel service.
   */
  public static function logger() {
    return \Drupal::logger('ssid');
  }

}
