<?php

namespace Drupal\ssid\Service;

use Drupal\Core\Database\Connection;
use Drupal\Core\Lock\LockBackendInterface;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Entity\EntityFieldManagerInterface;
use Drupal\Core\Field\FieldDefinitionInterface;
use Psr\Log\LoggerInterface;
use Drupal\ssid\ScopePluginManagerInterface;

/**
 * Service to generate sequential serial numbers by scope and fields of type "scope_serial".
 */
final class SerialHandler implements SerialHandlerInterface {

  protected Connection $connection;
  protected LockBackendInterface $lock;
  protected EntityFieldManagerInterface $fieldManager;
  protected ScopePluginManagerInterface $scopePluginManager;
  protected LoggerInterface $logger;

  /**
   * Class constructor.
   *
   * @param \Drupal\Core\Database\Connection $connection
   *   The database connection service.
   * @param \Drupal\Core\Lock\LockBackendInterface $lock
   *   The lock backednd interface.
   * @param \Drupal\Core\Entity\EntityFieldManagerInterface $field_manager
   *   The field manager instance.
   * @param \Drupal\ssid\ScopePluginManagerInterface $scope_plugin_manager
   *   The scope plugin manager.
   * @param \Psr\Log\LoggerInterface $logger
   *   The logger channel instance.
   */
  public function __construct(
    Connection $connection,
    LockBackendInterface $lock,
    EntityFieldManagerInterface $field_manager,
    ScopePluginManagerInterface $scope_plugin_manager,
    LoggerInterface $logger
  ) {
    $this->connection = $connection;
    $this->lock = $lock;
    $this->fieldManager = $field_manager;
    $this->scopePluginManager = $scope_plugin_manager;
    $this->logger = $logger;
  }

  /**
   * {@inheritdoc}
   */
  public function getScopeDefinitionNames() : array {
    $options = [];
    $definitions = $this->scopePluginManager->getDefinitions();
    foreach ($definitions as $plugin_id => $definition) {
      $options[$plugin_id] = $definition['label'];
    }

    return $options;
  }

  /**
   * {@inheritdoc}
   */
  public function generateSerial(EntityInterface $entity, string $field_name) : array {
    $data = [];
    $entity_type_id = $entity->getEntityTypeId();
    $bundle = $entity->bundle();
    $field_definition = $entity->getFieldDefinition($field_name);
    if (!$field_definition instanceof FieldDefinitionInterface || $field_definition->getType() != 'scope_serial') {
      return $data;
    }
    $table = "{$entity_type_id}__{$field_name}";
    $scope_column = "{$field_name}_scope";
    $serial_column = "{$field_name}_serial";
    // Now, we need to get the scope according to configured plugin.
    $scope_plugin_id = $field_definition->getSetting('scope_plugin');
    $scope_plugin_instance = $this->scopePluginManager->createInstance($scope_plugin_id);
    $scope = $scope_plugin_instance->getScope($entity, ['field_name' => $field_name]);
    // To make this a safe query.
    $lock_name = "ssid:{$entity_type_id}:{$bundle}:{$field_name}:{$scope}";
    if (!$this->lock->acquire($lock_name, 5.0)) {
      $this->logger->warning('Lock timeout while generating serial for {entity_type}.{bundle}.{field} ({scope})', [
        'entity_type' => $entity_type_id,
        'bundle' => $bundle,
        'field' => $field_name,
        'scope' => $scope,
      ]);
      throw new \RuntimeException("Unable to acquire lock for serial generation: {$lock_name}");
    }
    try {
      $transaction = $this->connection->startTransaction();
      $query = $this->connection->select($table, 't')
        ->fields('t', [$serial_column])
        ->condition($scope_column, $scope);
      if ((bool) $field_definition->getSetting('bundle')) {
        $query->condition('bundle', $bundle);
      }
      $last_serial = $query
        ->orderBy($serial_column, 'DESC')
        ->range(0, 1)
        ->execute()
        ->fetchField();
      $serial = ((int) $last_serial) + 1;
      $data = [
        'plugin' => $scope_plugin_id,
        'scope' => $scope,
        'serial' => $serial,
      ];
      unset($transaction);
    }
    catch (\Throwable $e) {
      // Rollback handled automatically by Drupal transaction, but log error.
      $this->logger->error('Serial generation failed for {entity_type}.{field} ({scope}): {message}', [
        'entity_type' => $entity_type_id,
        'field' => $field_name,
        'scope' => $scope,
        'message' => $e->getMessage(),
      ]);

      throw $e;
    }
    finally {
      $this->lock->release($lock_name);
    }

    return $data;
  }

}
