<?php

namespace Drupal\sso_bouncer\Drush\Commands;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Utility\Token;
use Drush\Attributes as CLI;
use Drush\Commands\AutowireTrait;
use Drush\Commands\DrushCommands;

/**
 * A Drush commandfile for SSO Bouncer operations.
 */
final class SsoBouncerCommands extends DrushCommands {

  use AutowireTrait;

  /**
   * Constructs a SsoBouncerCommands object.
   */
  public function __construct(
    private readonly Token $token,
    private readonly ConfigFactoryInterface $configFactory,
  ) {
    parent::__construct();
  }

  /**
   * Enable SSO Bouncer with a client ID.
   */
  #[CLI\Command(name: 'sso_bouncer:enable', aliases: ['ssoe'])]
  #[CLI\Argument(name: 'clientId', description: 'The Keycloak client ID to use for SSO authentication.')]
  #[CLI\Usage(name: 'sso_bouncer:enable my-client-id', description: 'Enable SSO Bouncer with client ID "my-client-id"')]
  public function enable($clientId) {
    $config = $this->configFactory->getEditable('sso_bouncer.settings');

    if (empty($clientId)) {
      $this->logger()->error(dt('Client ID is required to enable SSO Bouncer.'));
      return;
    }

    $config->set('enabled', TRUE);
    $config->set('clientId', $clientId);
    $config->save();

    $this->logger()->success(dt('SSO Bouncer has been enabled with client ID: @clientId', ['@clientId' => $clientId]));
  }

  /**
   * Disable SSO Bouncer.
   */
  #[CLI\Command(name: 'sso_bouncer:disable', aliases: ['ssod'])]
  #[CLI\Usage(name: 'sso_bouncer:disable', description: 'Disable SSO Bouncer authentication')]
  public function disable() {
    $config = $this->configFactory->getEditable('sso_bouncer.settings');

    $config->set('enabled', FALSE);
    $config->save();

    $this->logger()->success(dt('SSO Bouncer has been disabled.'));
  }

  /**
   * Show current SSO Bouncer status and configuration.
   */
  #[CLI\Command(name: 'sso_bouncer:status', aliases: ['ssos'])]
  #[CLI\Usage(name: 'sso_bouncer:status', description: 'Show current SSO Bouncer configuration')]
  public function status() {
    $config = $this->configFactory->get('sso_bouncer.settings');

    $enabled = $config->get('enabled') ?? FALSE;
    $clientId = $config->get('clientId') ?? '';

    $this->output()->writeln('SSO Bouncer Status:');
    $this->output()->writeln('  Enabled: ' . ($enabled ? 'Yes' : 'No'));
    $this->output()->writeln('  Client ID: ' . ($clientId ?: 'Not set'));
  }

}
