<?php

declare(strict_types=1);

namespace Drupal\state_machine_automated_transition\Form;

use Drupal\Component\Render\FormattableMarkup;
use Drupal\Component\Utility\NestedArray;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\state_machine\WorkflowManager;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Configure State Machine Automated Transition settings for this site.
 */
final class SettingsForm extends ConfigFormBase {

  /**
   * The workflow manager.
   *
   * @var \Drupal\state_machine\WorkflowManager
   */
  protected WorkflowManager $workflowManager;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    $class = parent::create($container);
    $class->workflowManager = $container->get('plugin.manager.workflow');
    return $class;
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'state_machine_automated_transition_settings';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames(): array {
    return ['state_machine_automated_transition.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    $groups = $this->workflowManager->getGroupedLabels();
    $values = $this->config('state_machine_automated_transition.settings')->get('allow_list') ?? [];
    $form['help']['#markup'] = '<p>' . $this->t('Select transitions that should be allowed to be automatically applied. Please note that custom automators can override this setting.') . '</p>';
    $form['allow_list']['#tree'] = TRUE;
    foreach ($groups as $group_label => $group) {
      foreach ($group as $definition_id => $label) {
        $definition = $this->workflowManager->getDefinition($definition_id);
        if (!$definition) {
          continue;
        }
        $group_id = $definition['group'];
        $labels = $definition['transitions'];
        $labels = array_map(function (array $value) {
          return $value['label'];
        }, $labels);

        $form['allow_list'][$group_id][$definition_id] = [
          '#type' => 'checkboxes',
          '#multiple' => TRUE,
          '#title' => new FormattableMarkup(":workflow_group_label / :workflow_label", [
            ':workflow_group_label' => $group_label,
            ':workflow_label' => $label,
          ]),
          '#default_value' => $values[$group_id][$definition_id] ?? [],
          '#options' => $labels,
        ];
      }
    }
    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    $allow_list = $form_state->getValue('allow_list');
    $allow_list = NestedArray::filter($allow_list);
    $this->config('state_machine_automated_transition.settings')
      ->set('allow_list', $allow_list)
      ->save();
    parent::submitForm($form, $form_state);
  }

}
