<?php

namespace Drupal\state_machine_automated_transition;

use Drupal\Core\Entity\EntityInterface;
use Drupal\state_machine\Plugin\Workflow\WorkflowInterface;
use Drupal\state_machine\Plugin\Workflow\WorkflowTransition;
use Symfony\Component\DependencyInjection\Attribute\AutowireIterator;

/**
 * Manages transition automators within a workflow system.
 */
final class TransitionAutomatorManager {

  /**
   * Constructor of the automator manager.
   *
   * @param iterable<\Drupal\state_machine_automated_transition\TransitionAutomatorInterface> $transitionAutomators
   *   Transition automators.
   */
  public function __construct(
    #[AutowireIterator('state_machine_automated_transition.transition_automator')]
    protected iterable $transitionAutomators,
  ) {}

  /**
   * Determines if the transition can be applied based on the automators.
   *
   * @param \Drupal\state_machine\Plugin\Workflow\WorkflowTransition $transition
   *   The workflow transition to be evaluated.
   * @param \Drupal\state_machine\Plugin\Workflow\WorkflowInterface $workflow
   *   The workflow to which the transition belongs.
   * @param \Drupal\Core\Entity\EntityInterface $entity
   *   The entity on which the workflow transition is being applied.
   *
   * @return \Drupal\state_machine_automated_transition\TransitionAutomatorResult
   *   The result indicating whether the transition is allowed or forbidden.
   */
  public function applies(WorkflowTransition $transition, WorkflowInterface $workflow, EntityInterface $entity) : TransitionAutomatorResult {
    $allowed = FALSE;
    foreach ($this->transitionAutomators as $automator) {
      $result = $automator->applies($transition, $workflow, $entity);
      if ($result->isForbidden()) {
        return $result;
      }
      if ($result->isAllowed()) {
        $allowed = TRUE;
      }
    }
    return $allowed
      ? TransitionAutomatorResult::allowed()
      : TransitionAutomatorResult::forbidden('There are no automators allowing this transition.');
  }

}
