<?php

namespace Drupal\state_machine_automated_transition;

use Drupal\Core\DestructableInterface;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Entity\FieldableEntityInterface;
use Drupal\state_machine\Plugin\Workflow\WorkflowTransition;

/**
 * Class responsible for orchestrating automated state transitions.
 *
 * This class manages a queue of state transitions for entities and processes
 * them in a first-come first-served order. It interacts with the entity type
 * manager to retrieve storage and apply transitions.
 */
final class TransitionAutomatorOrchestrator implements DestructableInterface {

  /**
   * List of transitions.
   *
   * @var array
   */
  protected $transitions = [];

  public function __construct(
    protected EntityTypeManagerInterface $entityTypeManager,
  ) {}

  /**
   * {@inheritdoc}
   */
  public function destruct() {
    foreach ($this->transitions as $entity_type => $transitions) {
      $entity_storage[$entity_type] = $this->entityTypeManager->getStorage($entity_type);
      while (count($this->transitions[$entity_type])) {
        $transition = array_shift($this->transitions[$entity_type]);
        $entity_id = $transition['entity_id'];
        $entity = $entity_storage[$entity_type]->loadUnchanged($entity_id);
        // Skip if the entity cannot be loaded.
        // This is a fail-safe to ensure the entity supports fields before
        // attempting to access them.
        if (!$entity instanceof FieldableEntityInterface) {
          continue;
        }
        /** @var \Drupal\state_machine\Plugin\Field\FieldType\StateItemInterface $state */
        $state = $entity->get($transition['field_name'])->first();
        $state->applyTransitionById($transition['transition_id']);
        $entity->save();
      }
    }
  }

  /**
   * Adds transition to queue.
   *
   * Transitions will be processed in first-come first-served order.
   *
   * @param \Drupal\Core\Entity\EntityInterface $entity
   *   The entity.
   * @param string $field_name
   *   The field name of the state type.
   * @param \Drupal\state_machine\Plugin\Workflow\WorkflowTransition $transition
   *   The transition to apply.
   */
  public function addTransition(EntityInterface $entity, string $field_name, WorkflowTransition $transition): void {
    $this->transitions[$entity->getEntityTypeId()][$entity->id()] = [
      'entity_id' => $entity->id(),
      'field_name' => $field_name,
      'transition_id' => $transition->getId(),
    ];
  }

}
