<?php

namespace Drupal\state_machine_automated_transition;

use Drupal\Component\Render\MarkupInterface;

/**
 * Provides a value object representing the automator status.
 */
final class TransitionAutomatorResult {

  /**
   * The automator result.
   *
   * @var bool|null
   */
  protected $result;

  /**
   * The automator result "reason".
   *
   * @var mixed|null
   */
  protected $reason;

  /**
   * The automator result "code".
   *
   * @var mixed|null
   */
  protected $code;

  /**
   * Constructs a new AutomatorResult object.
   *
   * @param bool|null $result
   *   The automator result, FALSE when forbidden, TRUE when allowed.
   * @param mixed|null $reason
   *   (optional) The reason why an order item is forbidden.
   * @param mixed|null $code
   *   (optional) The "code" explaining why an order item is forbidden.
   */
  public function __construct(?bool $result, $reason = NULL, $code = NULL) {
    assert(is_string($reason) || is_null($reason) || $reason instanceof MarkupInterface);
    assert(is_string($code) || is_null($code) || $code instanceof MarkupInterface);
    $this->result = $result;
    $this->reason = $reason;
    $this->code = $code;
  }

  /**
   * Creates an automator result that is "allowed".
   *
   * @return static
   */
  public static function allowed() : TransitionAutomatorResult {
    return new static(TRUE);
  }

  /**
   * Creates an automator result that is "neutral".
   *
   * @return static
   */
  public static function neutral() : TransitionAutomatorResult {
    return new static(NULL);
  }

  /**
   * Creates an automator result that is "forbidden".
   *
   * @param mixed|null $reason
   *   (optional) The reason why an order item is forbidden.
   * @param mixed|null $code
   *   (optional) The "code" explaining why an order item is forbidden.
   *
   * @return static
   */
  public static function forbidden($reason = NULL, $code = NULL) : TransitionAutomatorResult {
    return new static(FALSE, $reason, $code);
  }

  /**
   * Gets the "code".
   *
   * @return mixed|null
   *   The "code" for this automator result, NULL when not provided.
   */
  public function getCode() {
    return $this->code;
  }

  /**
   * Gets the "reason".
   *
   * @return mixed|null
   *   The "reason" for this automator result, NULL when not provided.
   */
  public function getReason() {
    return $this->reason;
  }

  /**
   * Determines whether the automator result is "neutral".
   *
   * @return bool
   *   Whether the automator result is "neutral".
   */
  public function isNeutral(): bool {
    return $this->result === NULL;
  }

  /**
   * Determines whether the automator result is "forbidden".
   *
   * @return bool
   *   Whether the automator is "forbidden".
   */
  public function isForbidden(): bool {
    return $this->result === FALSE;
  }

  /**
   * Determines whether the automator result is "allowed".
   *
   * @return bool
   *   Whether the automator is "allowed".
   */
  public function isAllowed(): bool {
    return $this->result === TRUE;
  }

}
