/**
 * File Editor JavaScript
 */

(function($, Drupal) {
  'use strict';

  Drupal.behaviors.staticContentFileEditor = {
    attach: function(context, settings) {
      var $context = $(context);
      
      // Initialize full-screen toggle
      this.initFullScreen($context);
      
      // Initialize auto-save
      this.initAutoSave($context);
      
      // Initialize keyboard shortcuts
      this.initKeyboardShortcuts($context);
    },

    initFullScreen: function($context) {
      var $expandButton = $context.find('.scb-expand-button').once('scb-expand');
      if (!$expandButton.length) return;

      $expandButton.on('click', function(e) {
        e.preventDefault();
        Drupal.staticContentFileEditor.toggleFullScreen();
      });

      // ESC key to exit full screen
      $(document).on('keydown', function(e) {
        if (e.key === 'Escape' && $('.scb-fullscreen').length) {
          Drupal.staticContentFileEditor.toggleFullScreen();
        }
      });
    },

    initAutoSave: function($context) {
      var $form = $context.find('.scb-editor-form').once('scb-autosave');
      if (!$form.length) return;

      var saveTimeout;
      var $textarea = $form.find('textarea, .form-textarea');
      var $saveStatus = $form.find('.scb-save-status');

      $textarea.on('input', function() {
        clearTimeout(saveTimeout);
        $saveStatus.text('Unsaved changes').removeClass('saved error').addClass('saving');
        
        // Auto-save after 3 seconds of inactivity
        saveTimeout = setTimeout(function() {
          Drupal.staticContentFileEditor.autoSave($form);
        }, 3000);
      });
    },

    initKeyboardShortcuts: function($context) {
      var $form = $context.find('.scb-editor-form').once('scb-shortcuts');
      if (!$form.length) return;

      $(document).on('keydown', function(e) {
        // Ctrl+S or Cmd+S to save
        if ((e.ctrlKey || e.metaKey) && e.key === 's') {
          e.preventDefault();
          $form.find('input[type="submit"]').first().click();
        }
        
        // F11 for full screen
        if (e.key === 'F11') {
          e.preventDefault();
          Drupal.staticContentFileEditor.toggleFullScreen();
        }
      });
    }
  };

  // File Editor namespace
  Drupal.staticContentFileEditor = {
    
    toggleFullScreen: function() {
      var $editor = $('.scb-file-editor');
      
      if ($editor.hasClass('scb-fullscreen')) {
        // Exit full screen
        $editor.removeClass('scb-fullscreen');
        $('body').css('overflow', '');
        
        // Update expand button icon
        $('.scb-expand-button svg').html(this.getExpandIcon());
      } else {
        // Enter full screen
        $editor.addClass('scb-fullscreen');
        $('body').css('overflow', 'hidden');
        
        // Update expand button icon
        $('.scb-expand-button svg').html(this.getCollapseIcon());
        
        // Focus textarea
        $editor.find('textarea, .form-textarea').focus();
      }
    },

    autoSave: function($form) {
      var $saveStatus = $form.find('.scb-save-status');
      $saveStatus.text('Saving...').removeClass('saved error').addClass('saving');

      // Get form data
      var formData = new FormData($form[0]);
      
      $.ajax({
        url: $form.attr('action') || window.location.href,
        type: 'POST',
        data: formData,
        processData: false,
        contentType: false,
        success: function(response) {
          $saveStatus.text('Auto-saved').removeClass('saving error').addClass('saved');
        },
        error: function() {
          $saveStatus.text('Save failed').removeClass('saving saved').addClass('error');
        }
      });
    },

    addLineNumbers: function($textarea) {
      if ($textarea.siblings('.scb-line-numbers').length) return;
      
      var $container = $('<div class="scb-editor-with-numbers">');
      var $lineNumbers = $('<div class="scb-line-numbers">');
      
      $textarea.wrap($container);
      $textarea.before($lineNumbers);
      
      this.updateLineNumbers($textarea, $lineNumbers);
      
      $textarea.on('input scroll', function() {
        Drupal.staticContentFileEditor.updateLineNumbers($textarea, $lineNumbers);
      });
    },

    updateLineNumbers: function($textarea, $lineNumbers) {
      var lines = $textarea.val().split('\n').length;
      var lineNumberText = '';
      
      for (var i = 1; i <= lines; i++) {
        lineNumberText += i + '\n';
      }
      
      $lineNumbers.text(lineNumberText);
      $lineNumbers.scrollTop($textarea.scrollTop());
    },

    insertAtCursor: function($textarea, text) {
      var textarea = $textarea[0];
      var start = textarea.selectionStart;
      var end = textarea.selectionEnd;
      var value = textarea.value;
      
      textarea.value = value.substring(0, start) + text + value.substring(end);
      textarea.selectionStart = textarea.selectionEnd = start + text.length;
      
      $textarea.trigger('input');
    },

    getExpandIcon: function() {
      return '<path d="M8 3a.5.5 0 0 1 .5.5v2a.5.5 0 0 1-.5.5H6a.5.5 0 0 1 0-1h1.5V3.5A.5.5 0 0 1 8 3zM3.5 8a.5.5 0 0 1 .5-.5h2a.5.5 0 0 1 0 1H4.5v1.5a.5.5 0 0 1-1 0V8zM8 13a.5.5 0 0 1-.5-.5v-2a.5.5 0 0 1 .5-.5h2a.5.5 0 0 1 0 1H8.5v1.5a.5.5 0 0 1-.5.5zM12.5 8a.5.5 0 0 1-.5.5h-2a.5.5 0 0 1 0-1h1.5V6a.5.5 0 0 1 1 0v2z"/>';
    },

    getCollapseIcon: function() {
      return '<path d="M5.5 0a.5.5 0 0 1 .5.5v4A1.5 1.5 0 0 1 4.5 6h-4a.5.5 0 0 1 0-1h4a.5.5 0 0 0 .5-.5v-4a.5.5 0 0 1 .5-.5zM10.5 0a.5.5 0 0 1 .5.5v4a.5.5 0 0 0 .5.5h4a.5.5 0 0 1 0 1h-4A1.5 1.5 0 0 1 10 4.5v-4a.5.5 0 0 1 .5-.5zM10.5 16a.5.5 0 0 1-.5-.5v-4a.5.5 0 0 0-.5-.5h-4a.5.5 0 0 1 0-1h4a1.5 1.5 0 0 1 1.5 1.5v4a.5.5 0 0 1-.5.5zM5.5 16a.5.5 0 0 1-.5-.5v-4A1.5 1.5 0 0 1 6.5 10h4a.5.5 0 0 1 0 1h-4a.5.5 0 0 0-.5.5v4a.5.5 0 0 1-.5.5z"/>';
    }
  };

})(jQuery, Drupal);