/**
 * Static Content Browser JavaScript
 */

(function($, Drupal, once) {
  'use strict';

  Drupal.behaviors.staticContentBrowser = {
    attach: function(context, settings) {
      // Check if we need to open an editor after page reload.
      var editorUrl = sessionStorage.getItem('scb_open_editor');
      if (editorUrl) {
        sessionStorage.removeItem('scb_open_editor');
        window.open(editorUrl, '_blank');
      }

      // Initialize upload dropzone
      this.initDropzone(context, settings);

      // Initialize action menus
      this.initActionMenus(context, settings);

      // Initialize action buttons
      this.initActionButtons(context, settings);
    },

    initDropzone: function(context, settings) {
      var dropzoneElements = once('scb-dropzone', '.scb-dropzone', context);
      var $dropzone = $(dropzoneElements);
      
      if (!$dropzone.length) return;

      var browserSettings = drupalSettings.staticContentBrowser || {};
      
      $dropzone.on('dragover dragenter', function(e) {
        e.preventDefault();
        e.stopPropagation();
        $(this).addClass('dragover');
      });

      $dropzone.on('dragleave dragend', function(e) {
        e.preventDefault();
        e.stopPropagation();
        $(this).removeClass('dragover');
      });

      $dropzone.on('drop', function(e) {
        e.preventDefault();
        e.stopPropagation();
        $(this).removeClass('dragover');
        
        var files = e.originalEvent.dataTransfer.files;
        if (files.length > 0) {
          Drupal.staticContentBrowser.uploadFiles(files, browserSettings);
        }
      });

      $dropzone.on('click', function() {
        var $input = $('<input type="file" multiple style="display: none;">');
        $input.on('change', function() {
          if (this.files.length > 0) {
            Drupal.staticContentBrowser.uploadFiles(this.files, browserSettings);
          }
        });
        $input.click();
      });
    },

    initActionMenus: function(context, settings) {
      var triggerElements = once('scb-actions', '.scb-actions-trigger', context);
      var $triggers = $(triggerElements);
      
      $triggers.on('click', function(e) {
        e.preventDefault();
        e.stopPropagation();
        
        var $menu = $(this).siblings('.scb-actions-dropdown');
        
        // Close other menus
        $('.scb-actions-dropdown.show').removeClass('show');
        
        // Toggle this menu
        $menu.toggleClass('show');
      });

      // Close menus when clicking outside
      $(document).on('click', function() {
        $('.scb-actions-dropdown.show').removeClass('show');
      });

      // Handle menu actions (only for links with data-action attribute).
      var actionElements = once('scb-action', '.scb-actions-dropdown a[data-action]', context);
      var $actions = $(actionElements);

      $actions.on('click', function(e) {
        e.preventDefault();

        var action = $(this).data('action');
        var itemName = $(this).data('item');
        var browserSettings = drupalSettings.staticContentBrowser || {};

        Drupal.staticContentBrowser.handleAction(action, itemName, browserSettings);

        // Close menu.
        $(this).closest('.scb-actions-dropdown').removeClass('show');
      });

      // Close menu when clicking direct links (Download, Edit).
      var directLinkElements = once('scb-direct-link', '.scb-actions-dropdown a:not([data-action])', context);
      $(directLinkElements).on('click', function() {
        $(this).closest('.scb-actions-dropdown').removeClass('show');
      });
    },

    initActionButtons: function(context, settings) {
      var browserSettings = drupalSettings.staticContentBrowser || {};
      
      // New directory button
      var newDirElements = once('scb-new-dir', '.scb-new-directory', context);
      var $newDir = $(newDirElements);
      
      $newDir.on('click', function(e) {
        e.preventDefault();
        Drupal.staticContentBrowser.createDirectory(browserSettings);
      });

      // New file button
      var newFileElements = once('scb-new-file', '.scb-new-file', context);
      var $newFile = $(newFileElements);
      
      $newFile.on('click', function(e) {
        e.preventDefault();
        Drupal.staticContentBrowser.createFile(browserSettings);
      });
    }
  };

  // Static Content Browser namespace
  Drupal.staticContentBrowser = {
    
    uploadFiles: function(files, settings) {
      var formData = new FormData();
      formData.append('action', 'upload');
      formData.append('directory_type', settings.directoryType || '');
      formData.append('path', settings.path || '');
      // Add CSRF token if available
      if (settings.csrfToken) {
        formData.append('csrf_token', settings.csrfToken);
      }
      for (var i = 0; i < files.length; i++) {
        formData.append('files[]', files[i]);
      }

      this.showMessage('Uploading files...', 'info');

      $.ajax({
        url: settings.ajaxUrl,
        type: 'POST',
        data: formData,
        processData: false,
        contentType: false,
        success: function(response) {
          if (response.success) {
            Drupal.staticContentBrowser.showMessage(response.message, 'success');
            location.reload(); // Reload to show new files
          } else {
            var errorMsg = response.error;
            if (response.details && response.details.length > 0) {
              errorMsg += ':\n' + response.details.join('\n');
            }
            Drupal.staticContentBrowser.showMessage(errorMsg, 'error');
          }
        },
        error: function() {
          Drupal.staticContentBrowser.showMessage('Upload failed', 'error');
        }
      });
    },

    createDirectory: function(settings) {
      var directoryName = prompt('Enter directory name:', 'untitled');
      if (!directoryName) return;

      this.ajaxAction('create_directory', {directory_name: directoryName}, settings);
    },

    createFile: function(settings) {
      var filename = prompt('Enter filename:', 'index.html');
      if (!filename) return;

      this.ajaxAction('create_file', {filename: filename}, settings);
    },

    handleAction: function(action, itemName, settings) {
      switch (action) {
        case 'delete':
          if (confirm('Are you sure you want to delete "' + itemName + '"?')) {
            this.ajaxAction('delete', {item_name: itemName}, settings);
          }
          break;
          
        case 'rename':
          var newName = prompt('Enter new name:', itemName);
          if (newName && newName !== itemName) {
            this.ajaxAction('rename', {old_name: itemName, new_name: newName}, settings);
          }
          break;
          
        case 'move':
          var target = prompt('Enter target directory (.. for parent):', '');
          if (target !== null) {
            this.ajaxAction('move', {item_name: itemName, target_directory: target}, settings);
          }
          break;
          
        case 'download_zip':
          this.ajaxAction('download_zip', {directory_name: itemName}, settings);
          break;
      }
    },

    ajaxAction: function(action, data, settings) {
      var requestData = {
        action: action,
        directory_type: settings.directoryType || '',
        path: settings.path || ''
      };
      // Add CSRF token if available
      if (settings.csrfToken) {
        requestData.csrf_token = settings.csrfToken;
      }
      $.extend(requestData, data);

      $.ajax({
        url: settings.ajaxUrl,
        type: 'POST',
        data: requestData,
        success: function(response) {
          if (response.success) {
            Drupal.staticContentBrowser.showMessage(response.message, 'success');

            if (action === 'download_zip' && response.download_url) {
              window.location.href = response.download_url;
            } else if (action === 'create_file' && response.edit_url) {
              // Store URL to open editor after reload.
              sessionStorage.setItem('scb_open_editor', response.edit_url);
              location.reload();
            } else {
              location.reload();
            }
          } else {
            Drupal.staticContentBrowser.showMessage(response.error, 'error');
          }
        },
        error: function() {
          Drupal.staticContentBrowser.showMessage('Action failed', 'error');
        }
      });
    },

    showMessage: function(message, type) {
      // Remove existing messages
      $('.scb-message').remove();
      
      var $message = $('<div class="scb-message scb-message-' + type + '">' + message + '</div>');
      $('.static-content-browser').prepend($message);
      
      // Auto-hide after 5 seconds for success messages
      if (type === 'success') {
        setTimeout(function() {
          $message.fadeOut();
        }, 5000);
      }
    },

    formatFileSize: function(bytes) {
      if (bytes === 0) return '0 Bytes';
      var k = 1024;
      var sizes = ['Bytes', 'KB', 'MB', 'GB'];
      var i = Math.floor(Math.log(bytes) / Math.log(k));
      return parseFloat((bytes / Math.pow(k, i)).toFixed(2)) + ' ' + sizes[i];
    },

    formatDate: function(timestamp) {
      var date = new Date(timestamp * 1000);
      return date.toLocaleDateString() + ' ' + date.toLocaleTimeString();
    }
  };

  // Add global dragover/drop handler to prevent default browser behavior
  $(document).on('dragover drop', function(e) {
    e.preventDefault();
  });

})(jQuery, Drupal, once);
