<?php

namespace Drupal\static_content_browser\Breadcrumb;

use Drupal\Core\Breadcrumb\Breadcrumb;
use Drupal\Core\Breadcrumb\BreadcrumbBuilderInterface;
use Drupal\Core\Link;
use Drupal\Core\Routing\RouteMatchInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\Core\Url;

/**
 * Breadcrumb builder for Static Content Browser.
 */
class StaticContentBrowserBreadcrumbBuilder implements BreadcrumbBuilderInterface {

  use StringTranslationTrait;

  /**
   * {@inheritdoc}
   */
  public function applies(RouteMatchInterface $route_match) {
    $route_name = $route_match->getRouteName();
    // Apply to all browser routes.
    return $route_name && strpos($route_name, 'static_content_browser.browser') === 0;
  }

  /**
   * {@inheritdoc}
   */
  public function build(RouteMatchInterface $route_match) {
    $breadcrumb = new Breadcrumb();
    $breadcrumb->addCacheContexts(['route']);

    // Add Home link.
    $breadcrumb->addLink(Link::createFromRoute($this->t('Home'), '<front>'));

    // Add Administration link.
    $breadcrumb->addLink(Link::createFromRoute($this->t('Administration'), 'system.admin'));

    // Add Content link.
    $breadcrumb->addLink(Link::createFromRoute($this->t('Content'), 'system.admin_content'));

    // Add Static Content Browser base link.
    $breadcrumb->addLink(Link::createFromRoute(
      $this->t('Static Content Browser'),
      'static_content_browser.browser'
    ));

    // Get the current path segments from the route parameters.
    $route_name = $route_match->getRouteName();
    $params = $route_match->getParameters()->all();

    // Extract directory type and path segments.
    $directory_type = $params['level0'] ?? NULL;
    $path_segments = [];

    if ($directory_type) {
      // Add the directory type link.
      $breadcrumb->addLink(Link::fromTextAndUrl(
        $this->formatDirectoryName($directory_type),
        Url::fromUri('internal:/admin/content/static-content-browser/' . $directory_type)
      ));

      // Collect path segments from level1 onwards.
      for ($i = 1; $i <= 6; $i++) {
        $level_key = 'level' . $i;
        if (isset($params[$level_key])) {
          $path_segments[] = $params[$level_key];
        }
      }

      // Build breadcrumb links for each path segment.
      $cumulative_path = $directory_type;
      foreach ($path_segments as $index => $segment) {
        $cumulative_path .= '/' . $segment;

        // Only add link if it's not the last segment (current page).
        if ($index < count($path_segments) - 1) {
          $breadcrumb->addLink(Link::fromTextAndUrl(
            $segment,
            Url::fromUri('internal:/admin/content/static-content-browser/' . $cumulative_path)
          ));
        }
        else {
          // Last segment - just show as text (current page).
          $breadcrumb->addLink(Link::createFromRoute($segment, '<none>'));
        }
      }
    }

    return $breadcrumb;
  }

  /**
   * Format directory name for display.
   *
   * @param string $directory_name
   *   The directory name (e.g., 'static-content-nodes').
   *
   * @return string
   *   Formatted name (e.g., 'Static Content Nodes').
   */
  protected function formatDirectoryName(string $directory_name): string {
    return ucwords(str_replace('-', ' ', $directory_name));
  }

}
