<?php

namespace Drupal\static_content_browser\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Url;
use Drupal\static_content_browser\Service\StaticContentBrowserService;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\File\UploadedFile;

/**
 * AJAX Controller for Static Content Browser operations.
 */
class StaticContentBrowserAjaxController extends ControllerBase {

  /**
   * The static content browser service.
   *
   * @var \Drupal\static_content_browser\Service\StaticContentBrowserService
   */
  protected $browserService;

  /**
   * Constructs a StaticContentBrowserAjaxController object.
   */
  public function __construct(StaticContentBrowserService $browser_service) {
    $this->browserService = $browser_service;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('static_content_browser.service')
    );
  }

  /**
   * Handle AJAX requests.
   */
  public function handle(Request $request) {
    // CSRF token check
    $token = $request->request->get('csrf_token');
    if (!$this->currentUser()->hasPermission('view static content browser')) {
      return new JsonResponse(['error' => 'Access denied'], 403);
    }
    // Validate CSRF token for all modifying actions
    $modifying_actions = ['upload', 'create_directory', 'create_file', 'delete', 'rename', 'move'];
    $action = $request->request->get('action');
    if (in_array($action, $modifying_actions)) {
      if (!$token || !\Drupal::service('csrf_token')->validate($token, 'static_content_browser_ajax')) {
        return new JsonResponse(['error' => 'Invalid CSRF token'], 403);
      }
    }

    $directory_type = $request->request->get('directory_type');
    $path = $request->request->get('path', '');

    switch ($action) {
      case 'upload':
        return $this->handleUpload($request, $directory_type, $path);
      
      case 'create_directory':
        return $this->handleCreateDirectory($request, $directory_type, $path);
      
      case 'create_file':
        return $this->handleCreateFile($request, $directory_type, $path);
      
      case 'delete':
        return $this->handleDelete($request, $directory_type, $path);
      
      case 'rename':
        return $this->handleRename($request, $directory_type, $path);
      
      case 'move':
        return $this->handleMove($request, $directory_type, $path);
      
      case 'download_zip':
        return $this->handleDownloadZip($request, $directory_type, $path);
      
      default:
        return new JsonResponse(['error' => 'Invalid action'], 400);
    }
  }

  /**
   * Handle file uploads.
   */
  protected function handleUpload(Request $request, $directory_type, $path) {
    if (!$this->currentUser()->hasPermission('edit static content files')) {
      return new JsonResponse(['error' => 'Access denied'], 403);
    }

    $uploaded_files = $request->files->get('files', []);
    if (empty($uploaded_files)) {
      return new JsonResponse(['error' => 'No files uploaded'], 400);
    }

    $target_path = $this->browserService->getStaticContentPath($directory_type, $path);
    $results = [];
    $errors = [];

    foreach ($uploaded_files as $uploaded_file) {
      /** @var UploadedFile $uploaded_file */
      $filename = $uploaded_file->getClientOriginalName();
      $size = $uploaded_file->getSize();
      
      // Validate file
      $validation = $this->browserService->validateFile($filename, $size);
      if ($validation !== TRUE) {
        $errors[] = $filename . ': ' . $validation;
        continue;
      }

      // Check for conflicts
      $target_file = $target_path . '/' . $filename;
      if (file_exists($target_file)) {
        $errors[] = $filename . ': File already exists';
        continue;
      }

      // Move uploaded file
      try {
        $uploaded_file->move($target_path, $filename);
        $results[] = $filename;
      } catch (\Exception $e) {
        $errors[] = $filename . ': Failed to upload';
      }
    }

    if (!empty($errors)) {
      return new JsonResponse([
        'error' => 'Upload failed',
        'details' => $errors,
        'uploaded' => $results,
      ], 400);
    }

    return new JsonResponse([
      'success' => TRUE,
      'uploaded' => $results,
      'message' => count($results) . ' file(s) uploaded successfully',
    ]);
  }

  /**
   * Handle directory creation.
   */
  protected function handleCreateDirectory(Request $request, $directory_type, $path) {
    if (!$this->currentUser()->hasPermission('edit static content files')) {
      return new JsonResponse(['error' => 'Access denied'], 403);
    }

    $directory_name = $request->request->get('directory_name');
    if (empty($directory_name)) {
      return new JsonResponse(['error' => 'Directory name required'], 400);
    }

    $target_path = $this->browserService->getStaticContentPath($directory_type, $path);
    $new_directory = $target_path . '/' . $directory_name;

    if (file_exists($new_directory)) {
      return new JsonResponse(['error' => 'Directory already exists'], 400);
    }

    if (mkdir($new_directory, 0755)) {
      return new JsonResponse([
        'success' => TRUE,
        'message' => 'Directory created successfully',
      ]);
    }

    return new JsonResponse(['error' => 'Failed to create directory'], 500);
  }

  /**
   * Handle file creation.
   */
  protected function handleCreateFile(Request $request, $directory_type, $path) {
    if (!$this->currentUser()->hasPermission('edit static content files')) {
      return new JsonResponse(['error' => 'Access denied'], 403);
    }

    $filename = $request->request->get('filename');
    if (empty($filename)) {
      return new JsonResponse(['error' => 'Filename required'], 400);
    }

    $target_path = $this->browserService->getStaticContentPath($directory_type, $path);
    $new_file = $target_path . '/' . $filename;

    if (file_exists($new_file)) {
      return new JsonResponse(['error' => 'File already exists'], 400);
    }

    // Create empty file with basic content
    $default_content = '';
    $extension = strtolower(pathinfo($filename, PATHINFO_EXTENSION));
    
    if ($extension === 'html') {
      $default_content = "<!DOCTYPE html>\n<html>\n<head>\n    <title>New Page</title>\n</head>\n<body>\n    <h1>New Page</h1>\n    <p>Content goes here...</p>\n</body>\n</html>";
    }

    if (file_put_contents($new_file, $default_content) !== FALSE) {
      return new JsonResponse([
        'success' => TRUE,
        'message' => 'File created successfully',
        'edit_url' => Url::fromRoute('static_content_browser.file_editor', [
          'directory_type' => $directory_type,
        ], [
          'query' => ['file_path' => $path ? $path . '/' . $filename : $filename],
        ])->toString(),
      ]);
    }

    return new JsonResponse(['error' => 'Failed to create file'], 500);
  }

  /**
   * Handle file/directory deletion.
   */
  protected function handleDelete(Request $request, $directory_type, $path) {
    if (!$this->currentUser()->hasPermission('edit static content files')) {
      return new JsonResponse(['error' => 'Access denied'], 403);
    }

    $item_name = $request->request->get('item_name');
    if (empty($item_name)) {
      return new JsonResponse(['error' => 'Item name required'], 400);
    }

    $target_path = $this->browserService->getStaticContentPath($directory_type, $path);
    $item_path = $target_path . '/' . $item_name;

    if (!file_exists($item_path)) {
      return new JsonResponse(['error' => 'Item not found'], 404);
    }

    if (is_dir($item_path)) {
      if ($this->deleteDirectory($item_path)) {
        return new JsonResponse([
          'success' => TRUE,
          'message' => 'Directory deleted successfully',
        ]);
      }
    } else {
      if (unlink($item_path)) {
        return new JsonResponse([
          'success' => TRUE,
          'message' => 'File deleted successfully',
        ]);
      }
    }

    return new JsonResponse(['error' => 'Failed to delete item'], 500);
  }

  /**
   * Handle renaming files/directories.
   */
  protected function handleRename(Request $request, $directory_type, $path) {
    if (!$this->currentUser()->hasPermission('edit static content files')) {
      return new JsonResponse(['error' => 'Access denied'], 403);
    }

    $old_name = $request->request->get('old_name');
    $new_name = $request->request->get('new_name');
    
    if (empty($old_name) || empty($new_name)) {
      return new JsonResponse(['error' => 'Old and new names required'], 400);
    }

    $target_path = $this->browserService->getStaticContentPath($directory_type, $path);
    $old_path = $target_path . '/' . $old_name;
    $new_path = $target_path . '/' . $new_name;

    if (!file_exists($old_path)) {
      return new JsonResponse(['error' => 'Item not found'], 404);
    }

    if (file_exists($new_path)) {
      return new JsonResponse(['error' => 'Target name already exists'], 400);
    }

    if (rename($old_path, $new_path)) {
      return new JsonResponse([
        'success' => TRUE,
        'message' => 'Item renamed successfully',
      ]);
    }

    return new JsonResponse(['error' => 'Failed to rename item'], 500);
  }

  /**
   * Handle moving files.
   */
  protected function handleMove(Request $request, $directory_type, $path) {
    if (!$this->currentUser()->hasPermission('edit static content files')) {
      return new JsonResponse(['error' => 'Access denied'], 403);
    }

    $item_name = $request->request->get('item_name');
    $target_directory = $request->request->get('target_directory');
    
    if (empty($item_name)) {
      return new JsonResponse(['error' => 'Item name required'], 400);
    }

    $source_path = $this->browserService->getStaticContentPath($directory_type, $path);
    $source_file = $source_path . '/' . $item_name;

    if (!file_exists($source_file)) {
      return new JsonResponse(['error' => 'Item not found'], 404);
    }

    // Determine target path
    if ($target_directory === '..') {
      // Move to parent directory
      $path_parts = explode('/', $path);
      array_pop($path_parts);
      $target_path = $this->browserService->getStaticContentPath($directory_type, implode('/', $path_parts));
    } else {
      // Move to subdirectory
      $target_path = $this->browserService->getStaticContentPath($directory_type, $path . '/' . $target_directory);
    }

    if (!is_dir($target_path)) {
      return new JsonResponse(['error' => 'Target directory not found'], 404);
    }

    $target_file = $target_path . '/' . $item_name;
    if (file_exists($target_file)) {
      return new JsonResponse(['error' => 'Target file already exists'], 400);
    }

    if (rename($source_file, $target_file)) {
      return new JsonResponse([
        'success' => TRUE,
        'message' => 'Item moved successfully',
      ]);
    }

    return new JsonResponse(['error' => 'Failed to move item'], 500);
  }

  /**
   * Handle ZIP download request.
   *
   * Returns a download URL that the client can use to download the ZIP.
   */
  protected function handleDownloadZip(Request $request, $directory_type, $path) {
    $directory_name = $request->request->get('directory_name');
    if (empty($directory_name)) {
      return new JsonResponse(['error' => 'Directory name required'], 400);
    }

    $source_path = $this->browserService->getStaticContentPath($directory_type, $path);
    $zip_source = $source_path . '/' . $directory_name;

    if (!is_dir($zip_source)) {
      return new JsonResponse(['error' => 'Directory not found'], 404);
    }

    // Generate a CSRF token for the download URL.
    $token = \Drupal::service('csrf_token')->get('static_content_browser_download');

    // Build the download URL with query parameters.
    $download_url = Url::fromRoute('static_content_browser.download_zip', [], [
      'query' => [
        'directory_type' => $directory_type,
        'path' => $path,
        'directory_name' => $directory_name,
        'token' => $token,
      ],
    ])->toString();

    return new JsonResponse([
      'success' => TRUE,
      'download_url' => $download_url,
    ]);
  }

  /**
   * Recursively delete directory.
   */
  protected function deleteDirectory($dir) {
    if (!is_dir($dir)) {
      return FALSE;
    }

    $files = array_diff(scandir($dir), ['.', '..']);
    foreach ($files as $file) {
      $path = $dir . '/' . $file;
      if (is_dir($path)) {
        $this->deleteDirectory($path);
      } else {
        unlink($path);
      }
    }

    return rmdir($dir);
  }

}