<?php

namespace Drupal\static_content_browser\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Url;
use Drupal\static_content_browser\Service\StaticContentBrowserService;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\StreamedResponse;

/**
 * Controller for Static Content Browser.
 */
class StaticContentBrowserController extends ControllerBase {

  /**
   * The static content browser service.
   *
   * @var \Drupal\static_content_browser\Service\StaticContentBrowserService
   */
  protected $browserService;

  /**
   * Constructs a StaticContentBrowserController object.
   */
  public function __construct(StaticContentBrowserService $browser_service) {
    $this->browserService = $browser_service;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('static_content_browser.service')
    );
  }

  /**
   * Main browser interface.
   *
   * @param string $directory_type
   *   The directory type (e.g., 'static-content-nodes').
   * @param string $path
   *   The path within the directory.
   *
   * @return array
   *   A render array.
   */
  public function browser(string $directory_type = '', string $path = ''): array {
    // Ensure main directories exist.
    $missing_directories = $this->browserService->ensureDirectoriesExist();

    // If no directory type specified, show top level.
    if (empty($directory_type)) {
      return $this->buildTopLevelBrowser($missing_directories);
    }

    // Build directory browser.
    return $this->buildDirectoryBrowser($directory_type, $path);
  }

  /**
   * Browser for the top level (no path segments).
   *
   * @return array
   *   A render array.
   */
  public function browser0(): array {
    return $this->browser('', '');
  }

  /**
   * Browser for 1 level deep paths (directory type only).
   *
   * @param string $level0
   *   The directory type.
   *
   * @return array
   *   A render array.
   */
  public function browser1(string $level0): array {
    return $this->browser($level0, '');
  }

  /**
   * Browser for 2 levels deep paths.
   *
   * @param string $level0
   *   The directory type.
   * @param string $level1
   *   The first path segment.
   *
   * @return array
   *   A render array.
   */
  public function browser2(string $level0, string $level1): array {
    return $this->browser($level0, $level1);
  }

  /**
   * Browser for 3 levels deep paths.
   *
   * @param string $level0
   *   The directory type.
   * @param string $level1
   *   The first path segment.
   * @param string $level2
   *   The second path segment.
   *
   * @return array
   *   A render array.
   */
  public function browser3(string $level0, string $level1, string $level2): array {
    $path = implode('/', [$level1, $level2]);
    return $this->browser($level0, $path);
  }

  /**
   * Browser for 4 levels deep paths.
   *
   * @param string $level0
   *   The directory type.
   * @param string $level1
   *   The first path segment.
   * @param string $level2
   *   The second path segment.
   * @param string $level3
   *   The third path segment.
   *
   * @return array
   *   A render array.
   */
  public function browser4(string $level0, string $level1, string $level2, string $level3): array {
    $path = implode('/', [$level1, $level2, $level3]);
    return $this->browser($level0, $path);
  }

  /**
   * Browser for 5 levels deep paths.
   *
   * @param string $level0
   *   The directory type.
   * @param string $level1
   *   The first path segment.
   * @param string $level2
   *   The second path segment.
   * @param string $level3
   *   The third path segment.
   * @param string $level4
   *   The fourth path segment.
   *
   * @return array
   *   A render array.
   */
  public function browser5(string $level0, string $level1, string $level2, string $level3, string $level4): array {
    $path = implode('/', [$level1, $level2, $level3, $level4]);
    return $this->browser($level0, $path);
  }

  /**
   * Browser for 6 levels deep paths.
   *
   * @param string $level0
   *   The directory type.
   * @param string $level1
   *   The first path segment.
   * @param string $level2
   *   The second path segment.
   * @param string $level3
   *   The third path segment.
   * @param string $level4
   *   The fourth path segment.
   * @param string $level5
   *   The fifth path segment.
   *
   * @return array
   *   A render array.
   */
  public function browser6(string $level0, string $level1, string $level2, string $level3, string $level4, string $level5): array {
    $path = implode('/', [$level1, $level2, $level3, $level4, $level5]);
    return $this->browser($level0, $path);
  }

  /**
   * Browser for 7 levels deep paths.
   *
   * @param string $level0
   *   The directory type.
   * @param string $level1
   *   The first path segment.
   * @param string $level2
   *   The second path segment.
   * @param string $level3
   *   The third path segment.
   * @param string $level4
   *   The fourth path segment.
   * @param string $level5
   *   The fifth path segment.
   * @param string $level6
   *   The sixth path segment.
   *
   * @return array
   *   A render array.
   */
  public function browser7(string $level0, string $level1, string $level2, string $level3, string $level4, string $level5, string $level6): array {
    $path = implode('/', [$level1, $level2, $level3, $level4, $level5, $level6]);
    return $this->browser($level0, $path);
  }

  /**
   * Build top-level browser showing the six main directories.
   */
  protected function buildTopLevelBrowser($missing_directories) {
    $allowed_directories = $this->browserService->getAllowedDirectories();
    
    $build = [
      '#theme' => 'static_content_browser_top_level',
      '#directories' => $allowed_directories,
      '#missing_directories' => $missing_directories,
      '#attached' => [
        'library' => ['static_content_browser/browser'],
      ],
    ];
  
    // Show suggestion to install Static Content Type module if directories are missing
    if (!empty($missing_directories)) {
      $this->messenger()->addWarning(
        $this->t('Some directories are missing. Consider installing the <a href="@url">Static Content Type module</a> to create them automatically.', [
          '@url' => 'https://www.drupal.org/project/static_content_type'
        ])
      );
    }

    return $build;
  }

  /**
   * Build directory browser for specific directory.
   */
  protected function buildDirectoryBrowser($directory_type, $path) {
    $items = $this->browserService->getDirectoryListing($directory_type, $path);
    
    if ($items === FALSE) {
      $this->messenger()->addError($this->t('Directory not found.'));
      return $this->redirect('static_content_browser.browser');
    }

    // Get navigation information
    $navigation = $this->browserService->getNavigationInfo($directory_type, $path);
    
    // Get active directory for index.html viewing
    $active_directory = $this->browserService->getActiveDirectory($directory_type, $path);
    $index_html_path = NULL;
    if ($active_directory !== NULL) {
      $base_path = '/sites/default/files/' . $directory_type;
      if ($path) {
        $base_path .= '/' . $path;
      }
      if ($active_directory !== '') {
        $index_html_path = $base_path . '/' . $active_directory . '/index.html';
      } else {
        $index_html_path = $base_path . '/index.html';
      }
    }
    
    // Build current path display
    $current_path = $this->browserService->getStaticContentPath($directory_type, $path);
    $public_path = $this->browserService->getPublicFilesPath();
    $display_path = str_replace($public_path, '/sites/default/files', $current_path);

    $build = [
      '#theme' => 'static_content_browser_directory',
      '#directory_type' => $directory_type,
      '#path' => $path,
      '#display_path' => $display_path,
      '#items' => $items,
      '#navigation' => $navigation,
      '#active_directory' => $active_directory,
      '#index_html_path' => $index_html_path,
      '#can_edit' => $this->currentUser()->hasPermission('edit static content files'),
      '#attached' => [
        'library' => ['static_content_browser/browser'],
        'drupalSettings' => [
          'staticContentBrowser' => [
            'directoryType' => $directory_type,
            'path' => $path,
            'ajaxUrl' => Url::fromRoute('static_content_browser.ajax')->toString(),
            // Add CSRF token for JS
            'csrfToken' => \Drupal::service('static_content_browser.token')->getToken(),
          ],
        ],
      ],
    ];

    return $build;
  }

  /**
   * File editor interface.
   */
  public function fileEditor($directory_type, Request $request) {

    $file_path = $request->query->get('file_path');
    
    if (!$file_path) {
      $this->messenger()->addError($this->t('No file specified.'));
      return $this->redirect('static_content_browser.browser1', ['level0' => $directory_type]);
    }

    // Continue with your existing fileEditor logic...
    if (!$this->currentUser()->hasPermission('edit static content files')) {
      throw new \Symfony\Component\HttpKernel\Exception\AccessDeniedHttpException();
    }

    if (!$this->currentUser()->hasPermission('edit static content files')) {
      throw new \Symfony\Component\HttpKernel\Exception\AccessDeniedHttpException();
    }

    $full_path = $this->browserService->getStaticContentPath($directory_type, $file_path);
    
    if (!file_exists($full_path) || !is_file($full_path)) {
      $this->messenger()->addError($this->t('File not found.'));
      return $this->redirect('static_content_browser.browser1', ['level0' => $directory_type]);
    }

    // Check if file is editable (text file)
    $extension = strtolower(pathinfo($file_path, PATHINFO_EXTENSION));
    $text_extensions = ['html', 'css', 'js', 'json', 'txt', 'md', 'yml', 'yaml'];
    
    if (!in_array($extension, $text_extensions)) {
      $this->messenger()->addError($this->t('This file type cannot be edited.'));
      return $this->redirect('static_content_browser.browser1', ['level0' => $directory_type]);
    }

    // Load file content
    $content = file_get_contents($full_path);
    
    // Build editor form
    $form = \Drupal::formBuilder()->getForm(
      'Drupal\static_content_browser\Form\FileEditorForm',
      $content,
      $file_path,
      $directory_type
    );

    return $form;
  }

  /**
   * Download a directory as a ZIP file.
   *
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The request object.
   *
   * @return \Symfony\Component\HttpFoundation\Response
   *   The ZIP file response.
   */
  public function downloadZip(Request $request): Response {
    $directory_type = $request->query->get('directory_type');
    $path = $request->query->get('path', '');
    $directory_name = $request->query->get('directory_name');
    $token = $request->query->get('token');

    // Validate CSRF token.
    if (!$token || !\Drupal::service('csrf_token')->validate($token, 'static_content_browser_download')) {
      return new Response('Access denied', 403);
    }

    // Validate required parameters.
    if (empty($directory_type) || empty($directory_name)) {
      return new Response('Invalid parameters', 400);
    }

    // Build the source path.
    $source_path = $this->browserService->getStaticContentPath($directory_type, $path);
    $zip_source = $source_path . '/' . $directory_name;

    if (!is_dir($zip_source)) {
      return new Response('Directory not found', 404);
    }

    // Generate filename.
    $filename = $directory_name . '_' . date('Y-m-d_H-i-s') . '.zip';

    // Create a streamed response that generates the ZIP on the fly.
    $response = new StreamedResponse(function () use ($zip_source, $directory_name) {
      // Create a temporary file for the ZIP.
      $temp_file = tempnam(sys_get_temp_dir(), 'scb_zip_');
      $zip = new \ZipArchive();

      if ($zip->open($temp_file, \ZipArchive::CREATE) === TRUE) {
        $this->addDirectoryToZip($zip, $zip_source, $directory_name);
        $zip->close();

        // Output the file.
        readfile($temp_file);

        // Clean up.
        unlink($temp_file);
      }
    });

    $response->headers->set('Content-Type', 'application/zip');
    $response->headers->set('Content-Disposition', 'attachment; filename="' . $filename . '"');

    return $response;
  }

  /**
   * Recursively add directory contents to a ZIP archive.
   *
   * @param \ZipArchive $zip
   *   The ZIP archive object.
   * @param string $source_dir
   *   The source directory path.
   * @param string $zip_path
   *   The path within the ZIP file.
   */
  protected function addDirectoryToZip(\ZipArchive $zip, string $source_dir, string $zip_path = ''): void {
    $files = array_diff(scandir($source_dir), ['.', '..']);

    foreach ($files as $file) {
      $file_path = $source_dir . '/' . $file;
      $zip_file_path = $zip_path ? $zip_path . '/' . $file : $file;

      if (is_dir($file_path)) {
        $zip->addEmptyDir($zip_file_path);
        $this->addDirectoryToZip($zip, $file_path, $zip_file_path);
      }
      else {
        $zip->addFile($file_path, $zip_file_path);
      }
    }
  }

}