<?php

namespace Drupal\static_content_browser\Form;

use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Url;

/**
 * File editor form for Static Content Browser.
 */
class FileEditorForm extends FormBase {

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'static_content_browser_file_editor_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state, $content = '', $file_path = '', $directory_type = '') {
    $config = $this->config('static_content_browser.settings');
    
    // Store form values for submit handler
    $form_state->set('file_path', $file_path);
    $form_state->set('directory_type', $directory_type);

    // Get file extension to determine editor behavior
    $extension = strtolower(pathinfo($file_path, PATHINFO_EXTENSION));
    
    // Display file path
    $form['file_info'] = [
      '#type' => 'item',
      '#title' => $this->t('Editing file'),
      '#markup' => '<strong>' . $file_path . '</strong>',
    ];

    // File content editor
    $form['content'] = [
      '#type' => 'text_format',
      '#title' => $this->t('File Content'),
      '#default_value' => $content,
      '#format' => $config->get('default_text_format') ?: 'full_html',
      '#rows' => 25,
      '#resizable' => TRUE,
    ];

    // For certain file types, use plain textarea for better editing
    if (in_array($extension, ['css', 'js', 'json', 'yml', 'yaml', 'txt', 'md'])) {
      $form['content'] = [
        '#type' => 'textarea',
        '#title' => $this->t('File Content'),
        '#default_value' => $content,
        '#rows' => 25,
        '#resizable' => TRUE,
        '#attributes' => [
          'style' => 'font-family: monospace; font-size: 14px;',
        ],
      ];
    }

    // Action buttons
    $form['actions'] = [
      '#type' => 'actions',
    ];

    $form['actions']['save'] = [
      '#type' => 'submit',
      '#value' => $this->t('Save File'),
      '#button_type' => 'primary',
    ];

    $form['actions']['save_close'] = [
      '#type' => 'submit',
      '#value' => $this->t('Save & Close'),
      '#submit' => ['::submitForm', '::submitSaveAndClose'],
    ];

    $form['actions']['cancel'] = [
      '#type' => 'link',
      '#title' => $this->t('Cancel'),
      '#url' => $this->getReturnUrl($directory_type, $file_path),
      '#attributes' => [
        'class' => ['button'],
      ],
    ];

    // Add JavaScript for full-screen functionality
    $form['#attached']['library'][] = 'static_content_browser/file_editor';

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $file_path = $form_state->get('file_path');
    $directory_type = $form_state->get('directory_type');
    
    // Get content - handle both text_format and textarea
    $content_value = $form_state->getValue('content');
    if (is_array($content_value) && isset($content_value['value'])) {
      // This is from text_format (HTML files)
      $content = $content_value['value'];
    } else {
      // This is from textarea (other file types)
      $content = $content_value;
    }

    // Get the full file path
    $browser_service = \Drupal::service('static_content_browser.service');
    $full_path = $browser_service->getStaticContentPath($directory_type, $file_path);

    // Backup original file
    $backup_path = $full_path . '.backup.' . time();
    if (file_exists($full_path)) {
      copy($full_path, $backup_path);
    }

    // Save new content
    if (file_put_contents($full_path, $content) !== FALSE) {
      $this->messenger()->addStatus($this->t('File saved successfully.'));
      // Remove backup on success
      if (file_exists($backup_path)) {
        unlink($backup_path);
      }
    } else {
      $this->messenger()->addError($this->t('Failed to save file.'));
      // Restore from backup if it exists
      if (file_exists($backup_path)) {
        copy($backup_path, $full_path);
        unlink($backup_path);
      }
    }
  }

  /**
   * Submit handler for save and close action.
   */
  public function submitSaveAndClose(array &$form, FormStateInterface $form_state) {
    $directory_type = $form_state->get('directory_type');
    $file_path = $form_state->get('file_path');
    
    $form_state->setRedirectUrl($this->getReturnUrl($directory_type, $file_path));
  }

  /**
   * Get the return URL for the browser.
   */
  protected function getReturnUrl($directory_type, $file_path) {
    // Extract directory path from file path.
    $path_parts = explode('/', $file_path);
    array_pop($path_parts);
    $directory_path = implode('/', $path_parts);

    // Build URL directly since routes use level-based parameters.
    $url_path = '/admin/content/static-content-browser/' . $directory_type;
    if ($directory_path) {
      $url_path .= '/' . $directory_path;
    }

    return Url::fromUserInput($url_path);
  }

}