<?php

namespace Drupal\static_content_browser\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;

/**
 * Configuration form for Static Content Browser settings.
 */
class StaticContentBrowserSettingsForm extends ConfigFormBase {

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return ['static_content_browser.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'static_content_browser_settings_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $config = $this->config('static_content_browser.settings');

    $form['file_settings'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('File Upload Settings'),
    ];

    $form['file_settings']['max_file_size'] = [
      '#type' => 'number',
      '#title' => $this->t('Maximum file size (bytes)'),
      '#default_value' => $config->get('max_file_size') ?: 10485760,
      '#description' => $this->t('Maximum file size for uploads. Default is 10MB (10485760 bytes).'),
      '#min' => 1,
      '#step' => 1,
    ];

    $form['file_settings']['allowed_extensions'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Allowed file extensions'),
      '#default_value' => $config->get('allowed_extensions') ?: 'jpg jpeg png gif svg webp html css js json txt md zip yml yaml',
      '#description' => $this->t('Space-separated list of allowed file extensions (without dots).'),
      '#size' => 80,
    ];

    $form['editor_settings'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Editor Settings'),
    ];

    $form['editor_settings']['default_text_format'] = [
      '#type' => 'select',
      '#title' => $this->t('Default text format'),
      '#options' => $this->getTextFormats(),
      '#default_value' => $config->get('default_text_format') ?: 'full_html',
      '#description' => $this->t('Text format to use for the file editor.'),
    ];

    $form['security_settings'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Security Settings'),
    ];

    $form['security_settings']['enable_path_validation'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable path validation'),
      '#default_value' => $config->get('enable_path_validation') !== FALSE,
      '#description' => $this->t('Enable strict path validation to prevent directory traversal attacks. Recommended to keep enabled.'),
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state) {
    $max_file_size = $form_state->getValue('max_file_size');
    if ($max_file_size < 1) {
      $form_state->setErrorByName('max_file_size', $this->t('Maximum file size must be at least 1 byte.'));
    }

    $allowed_extensions = $form_state->getValue('allowed_extensions');
    if (empty(trim($allowed_extensions))) {
      $form_state->setErrorByName('allowed_extensions', $this->t('At least one file extension must be allowed.'));
    }

    parent::validateForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $this->config('static_content_browser.settings')
      ->set('max_file_size', $form_state->getValue('max_file_size'))
      ->set('allowed_extensions', $form_state->getValue('allowed_extensions'))
      ->set('default_text_format', $form_state->getValue('default_text_format'))
      ->set('enable_path_validation', $form_state->getValue('enable_path_validation'))
      ->save();

    parent::submitForm($form, $form_state);
  }

  /**
   * Get available text formats.
   */
  protected function getTextFormats() {
    $formats = filter_formats();
    $options = [];
    foreach ($formats as $format_id => $format) {
      $options[$format_id] = $format->label();
    }
    return $options;
  }

}