<?php

namespace Drupal\static_content_browser\Service;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\File\FileSystemInterface;
use Drupal\Core\StreamWrapper\StreamWrapperManagerInterface;
use Drupal\Core\Messenger\MessengerInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;

/**
 * Service for managing static content browser operations.
 */
class StaticContentBrowserService {

  use StringTranslationTrait;

  /**
   * The file system service.
   *
   * @var \Drupal\Core\File\FileSystemInterface
   */
  protected $fileSystem;

  /**
   * The stream wrapper manager.
   *
   * @var \Drupal\Core\StreamWrapper\StreamWrapperManagerInterface
   */
  protected $streamWrapperManager;

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * The messenger service.
   *
   * @var \Drupal\Core\Messenger\MessengerInterface
   */
  protected $messenger;

  /**
   * Array of allowed static content directories.
   *
   * @var array
   */
  protected $allowedDirectories = [
    'static-content-nodes',
    'static-content-blocks', 
    'static-content-paragraphs',
    'static-content-sdc',
    'static-content-twig',
    'static-content-pages',
  ];

  /**
   * Array of allowed file extensions.
   *
   * @var array
   */
  protected $allowedExtensions = [
    'jpg', 'jpeg', 'png', 'gif', 'svg', 'webp',
    'html', 'css', 'js', 'json', 'txt', 'md',
    'zip', 'yml', 'yaml'
  ];

  /**
   * Directory precedence order for Static Content Type module.
   *
   * @var array
   */
  protected $directoryPrecedence = [
    'dist',
    'build', 
    'raw',
    'proxied',
    'hardened',
    'src'
  ];

  /**
   * Constructs a StaticContentBrowserService object.
   */
  public function __construct(
    FileSystemInterface $file_system,
    StreamWrapperManagerInterface $stream_wrapper_manager,
    ConfigFactoryInterface $config_factory,
    MessengerInterface $messenger
  ) {
    $this->fileSystem = $file_system;
    $this->streamWrapperManager = $stream_wrapper_manager;
    $this->configFactory = $config_factory;
    $this->messenger = $messenger;
  }

  /**
   * Get the public files directory path.
   */
  public function getPublicFilesPath() {
    return $this->fileSystem->realpath('public://');
  }

  /**
   * Get the full path for a static content directory.
   */
  public function getStaticContentPath($directory_type = '', $path = '') {
    $base_path = $this->getPublicFilesPath();
    
    if (empty($directory_type)) {
      return $base_path;
    }

    // Validate directory type
    if (!in_array($directory_type, $this->allowedDirectories)) {
      throw new \InvalidArgumentException("Invalid directory type: $directory_type");
    }

    $full_path = $base_path . '/' . $directory_type;
    
    if (!empty($path)) {
      // Sanitize path to prevent directory traversal
      $clean_path = $this->sanitizePath($path);
      $full_path .= '/' . $clean_path;
    }

    return $full_path;
  }

  /**
   * Sanitize file path to prevent directory traversal attacks.
   */
  protected function sanitizePath($path) {
    // Remove any dangerous characters and sequences
    $path = str_replace(['../', '..\\', '~/', './'], '', $path);
    $path = preg_replace('/[^a-zA-Z0-9\/_.-]/', '', $path);
    $path = trim($path, '/');
    
    return $path;
  }

  /**
   * Check if a directory exists and create the main directories if needed.
   */
  public function ensureDirectoriesExist() {
    $base_path = $this->getPublicFilesPath();
    $missing_directories = [];

    foreach ($this->allowedDirectories as $directory) {
      $dir_path = $base_path . '/' . $directory;
      if (!is_dir($dir_path)) {
        $missing_directories[] = $directory;
        // Try to create the directory
        if (!$this->fileSystem->mkdir($dir_path, 0755, TRUE)) {
          $this->messenger->addError($this->t('Could not create directory: @dir', ['@dir' => $directory]));
        }
      }
    }

    return $missing_directories;
  }

  /**
   * Get directory listing with file information.
   */
  public function getDirectoryListing($directory_type, $path = '') {
    $full_path = $this->getStaticContentPath($directory_type, $path);
    
    if (!is_dir($full_path)) {
      return FALSE;
    }

    $items = [];
    $files = scandir($full_path);
    
    foreach ($files as $file) {
      if ($file === '.' || $file === '..') {
        continue;
      }

      $file_path = $full_path . '/' . $file;
      $relative_path = $path ? $path . '/' . $file : $file;
      
      $item = [
        'name' => $file,
        'path' => $relative_path,
        'is_directory' => is_dir($file_path),
        'size' => is_file($file_path) ? filesize($file_path) : 0,
        'modified' => filemtime($file_path),
        'extension' => pathinfo($file, PATHINFO_EXTENSION),
      ];

      // Add precedence information for directories
      if ($item['is_directory'] && $this->isAtIndexLevel($directory_type, $path)) {
        $item['precedence_order'] = array_search($file, $this->directoryPrecedence);
        if ($item['precedence_order'] === FALSE) {
          $item['precedence_order'] = 999; // Not in precedence list
        }
      }

      $items[] = $item;
    }

    // Sort: directories first, then by precedence, then alphabetically
    usort($items, function($a, $b) {
      if ($a['is_directory'] && !$b['is_directory']) return -1;
      if (!$a['is_directory'] && $b['is_directory']) return 1;
      
      if ($a['is_directory'] && $b['is_directory']) {
        $precedence_a = $a['precedence_order'] ?? 999;
        $precedence_b = $b['precedence_order'] ?? 999;
        if ($precedence_a !== $precedence_b) {
          return $precedence_a <=> $precedence_b;
        }
      }
      
      return strcasecmp($a['name'], $b['name']);
    });

    return $items;
  }

  /**
   * Check if we're at the index level (directly under main directories).
   */
  protected function isAtIndexLevel($directory_type, $path) {
    // If path is empty, we're at the main directory level
    if (empty($path)) {
      return FALSE;
    }
    
    // If path has no slashes, we're at index level
    return strpos($path, '/') === FALSE;
  }

  /**
   * Get the active directory based on precedence rules.
   */
  public function getActiveDirectory($directory_type, $path) {
    if (!$this->isAtIndexLevel($directory_type, $path)) {
      return NULL;
    }

    $full_path = $this->getStaticContentPath($directory_type, $path);
    
    foreach ($this->directoryPrecedence as $precedence_dir) {
      $precedence_path = $full_path . '/' . $precedence_dir;
      if (is_dir($precedence_path) && file_exists($precedence_path . '/index.html')) {
        return $precedence_dir;
      }
    }

    // Check root level for index.html
    if (file_exists($full_path . '/index.html')) {
      return '';
    }

    return NULL;
  }

  /**
   * Validate file upload.
   */
  public function validateFile($filename, $size) {
    $config = $this->configFactory->get('static_content_browser.settings');
    $max_size = $config->get('max_file_size') ?: 10485760; // 10MB default
    
    // Check file size
    if ($size > $max_size) {
      return $this->t('File size (@size) exceeds maximum allowed size (@max)', [
        '@size' => format_size($size),
        '@max' => format_size($max_size),
      ]);
    }

    // Check file extension
    $extension = strtolower(pathinfo($filename, PATHINFO_EXTENSION));
    if (!in_array($extension, $this->allowedExtensions)) {
      return $this->t('File type "@ext" is not allowed', ['@ext' => $extension]);
    }

    return TRUE;
  }

  /**
   * Get navigation information for current directory.
   */
  public function getNavigationInfo($directory_type, $path) {
    $info = [
      'parent' => NULL,
      'siblings' => [],
      'current_index' => 0,
      'previous' => NULL,
      'next' => NULL,
    ];

    // Get parent directory
    if (!empty($path)) {
      $path_parts = explode('/', $path);
      array_pop($path_parts);
      $info['parent'] = implode('/', $path_parts);
    }

    // Get siblings at same level
    $parent_path = $info['parent'];
    $parent_full_path = $this->getStaticContentPath($directory_type, $parent_path);
    
    if (is_dir($parent_full_path)) {
      $siblings = [];
      $files = scandir($parent_full_path);
      
      foreach ($files as $file) {
        if ($file === '.' || $file === '..') {
          continue;
        }
        
        $file_path = $parent_full_path . '/' . $file;
        if (is_dir($file_path)) {
          $siblings[] = $parent_path ? $parent_path . '/' . $file : $file;
        }
      }
      
      sort($siblings);
      $info['siblings'] = $siblings;
      
      // Find current position
      $current_index = array_search($path, $siblings);
      if ($current_index !== FALSE) {
        $info['current_index'] = $current_index;
        $info['previous'] = $current_index > 0 ? $siblings[$current_index - 1] : NULL;
        $info['next'] = $current_index < count($siblings) - 1 ? $siblings[$current_index + 1] : NULL;
      }
    }

    return $info;
  }

  /**
   * Get allowed directories.
   */
  public function getAllowedDirectories() {
    return $this->allowedDirectories;
  }

  /**
   * Get allowed file extensions.
   */
  public function getAllowedExtensions() {
    return $this->allowedExtensions;
  }

}